<?php
/**
 * Plugin Name: Lean Bunker Sitemap
 * Description: Sitemap XML universale, leggera e scalabile. Sitemap index, post, pagine, tassonomie, immagini. Adattiva per siti piccoli e enormi. Zero dipendenze.
 * Version: 1.0.0
 * Author: Riccardo Bastillo
 * License: GPL-2.0+
 */

if (!defined('ABSPATH')) exit;

class Lean_Bunker_Sitemap {
    const OPTION_PREFIX = 'lb_sitemap_';
    const MAX_URLS_DEFAULT = 2000;

    public function __construct() {
        // Attivazione / disattivazione
        register_activation_hook(__FILE__, [$this, 'activate']);
        register_deactivation_hook(__FILE__, [$this, 'deactivate']);

        // Rewrite & query vars
        add_action('init', [$this, 'add_rewrite_rules']);
        add_filter('query_vars', [$this, 'add_query_vars']);

        // Template loader
        add_action('template_redirect', [$this, 'template_router'], 0);

        // Admin
        add_action('admin_menu', [$this, 'admin_menu']);

        // Robots.txt: aggiunge sitemap index
        add_filter('robots_txt', [$this, 'add_sitemap_to_robots'], 10, 2);
    }

    public function activate() {
        // Default options
        if (get_option(self::OPTION_PREFIX . 'max_urls') === false) {
            update_option(self::OPTION_PREFIX . 'max_urls', self::MAX_URLS_DEFAULT);
        }
        if (get_option(self::OPTION_PREFIX . 'split_by_date') === false) {
            update_option(self::OPTION_PREFIX . 'split_by_date', '1');
        }
        if (get_option(self::OPTION_PREFIX . 'include_images') === false) {
            update_option(self::OPTION_PREFIX . 'include_images', '1');
        }
        if (get_option(self::OPTION_PREFIX . 'include_taxonomies') === false) {
            update_option(self::OPTION_PREFIX . 'include_taxonomies', '1');
        }

        $this->add_rewrite_rules();
        flush_rewrite_rules();
    }

    public function deactivate() {
        flush_rewrite_rules();
    }

    public function add_query_vars($vars) {
        $vars[] = 'lb_sitemap';
        $vars[] = 'lb_sitemap_type';
        $vars[] = 'lb_sitemap_part';
        $vars[] = 'lb_sitemap_year';
        $vars[] = 'lb_sitemap_month';
        return $vars;
    }

    public function add_rewrite_rules() {
        // Index
        add_rewrite_rule('^sitemap-index\.xml$', 'index.php?lb_sitemap=1&lb_sitemap_type=index', 'top');

        // Posts numerate
        add_rewrite_rule('^sitemap-posts-([0-9]+)\.xml$', 'index.php?lb_sitemap=1&lb_sitemap_type=posts&lb_sitemap_part=$matches[1]', 'top');

        // Posts per anno/mese
        add_rewrite_rule('^sitemap-posts-([0-9]{4})-([0-9]{2})\.xml$', 'index.php?lb_sitemap=1&lb_sitemap_type=posts&lb_sitemap_year=$matches[1]&lb_sitemap_month=$matches[2]', 'top');

        // Pagine
        add_rewrite_rule('^sitemap-pages\.xml$', 'index.php?lb_sitemap=1&lb_sitemap_type=pages', 'top');

        // Tassonomie
        add_rewrite_rule('^sitemap-taxonomies\.xml$', 'index.php?lb_sitemap=1&lb_sitemap_type=taxonomies', 'top');

        // Immagini
        add_rewrite_rule('^sitemap-images-([0-9]+)\.xml$', 'index.php?lb_sitemap=1&lb_sitemap_type=images&lb_sitemap_part=$matches[1]', 'top');
    }

    public function template_router() {
        if (!get_query_var('lb_sitemap')) {
            return;
        }

        $type  = get_query_var('lb_sitemap_type');
        $part  = absint(get_query_var('lb_sitemap_part'));
        $year  = absint(get_query_var('lb_sitemap_year'));
        $month = absint(get_query_var('lb_sitemap_month'));

        // Header XML
        status_header(200);
        header('Content-Type: application/xml; charset=' . get_bloginfo('charset'), true);

        switch ($type) {
            case 'index':
                echo $this->render_index();
                break;
            case 'posts':
                echo $this->render_posts($part, $year, $month);
                break;
            case 'pages':
                echo $this->render_pages();
                break;
            case 'taxonomies':
                echo $this->render_taxonomies();
                break;
            case 'images':
                echo $this->render_images($part);
                break;
            default:
                status_header(404);
                echo '';
        }

        exit;
    }

    // ==========================
    // RENDER: SITEMAP INDEX
    // ==========================

    private function render_index() {
        $max_urls      = (int) get_option(self::OPTION_PREFIX . 'max_urls', self::MAX_URLS_DEFAULT);
        $split_by_date = get_option(self::OPTION_PREFIX . 'split_by_date', '1') === '1';
        $include_images = get_option(self::OPTION_PREFIX . 'include_images', '1') === '1';
        $include_tax   = get_option(self::OPTION_PREFIX . 'include_taxonomies', '1') === '1';

        $home = home_url('/');
        $now  = current_time('mysql');

        $xml  = '<?xml version="1.0" encoding="' . esc_attr(get_bloginfo('charset')) . '"?>' . "\n";
        $xml .= '<sitemapindex xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";

        // POSTS
        $post_counts = $this->get_post_counts_grouped($split_by_date);
        if ($split_by_date && !empty($post_counts['by_date'])) {
            foreach ($post_counts['by_date'] as $ym => $count) {
                list($y, $m) = explode('-', $ym);
                $loc = trailingslashit($home) . "sitemap-posts-{$y}-{$m}.xml";
                $xml .= '<sitemap><loc>' . esc_url($loc) . '</loc><lastmod>' . esc_html($now) . '</lastmod></sitemap>' . "\n";
            }
        } elseif (!empty($post_counts['total'])) {
            $total = (int) $post_counts['total'];
            $parts = max(1, ceil($total / $max_urls));
            for ($i = 1; $i <= $parts; $i++) {
                $loc = trailingslashit($home) . "sitemap-posts-{$i}.xml";
                $xml .= '<sitemap><loc>' . esc_url($loc) . '</loc><lastmod>' . esc_html($now) . '</lastmod></sitemap>' . "\n";
            }
        }

        // PAGES
        if ($this->has_pages()) {
            $loc = trailingslashit($home) . 'sitemap-pages.xml';
            $xml .= '<sitemap><loc>' . esc_url($loc) . '</loc><lastmod>' . esc_html($now) . '</lastmod></sitemap>' . "\n";
        }

        // TAXONOMIES
        if ($include_tax && $this->has_taxonomies()) {
            $loc = trailingslashit($home) . 'sitemap-taxonomies.xml';
            $xml .= '<sitemap><loc>' . esc_url($loc) . '</loc><lastmod>' . esc_html($now) . '</lastmod></sitemap>' . "\n";
        }

        // IMAGES (solo se ci sono allegati)
        if ($include_images && $this->has_attachments()) {
            $image_counts = $this->get_image_counts();
            $total_images = (int) $image_counts;
            if ($total_images > 0) {
                $parts = max(1, ceil($total_images / $max_urls));
                for ($i = 1; $i <= $parts; $i++) {
                    $loc = trailingslashit($home) . "sitemap-images-{$i}.xml";
                    $xml .= '<sitemap><loc>' . esc_url($loc) . '</loc><lastmod>' . esc_html($now) . '</lastmod></sitemap>' . "\n";
                }
            }
        }

        $xml .= '</sitemapindex>';
        return $xml;
    }

    // ==========================
    // RENDER: POSTS
    // ==========================

    private function render_posts($part = 1, $year = 0, $month = 0) {
        $max_urls = (int) get_option(self::OPTION_PREFIX . 'max_urls', self::MAX_URLS_DEFAULT);
        $part     = max(1, (int) $part);

        $args = [
            'post_type'      => 'post',
            'post_status'    => 'publish',
            'fields'         => 'ids',
            'posts_per_page' => $max_urls,
            'orderby'        => 'date',
            'order'          => 'DESC',
            'no_found_rows'  => true,
        ];

        if ($year > 0 && $month > 0) {
            $args['date_query'] = [
                [
                    'year'  => $year,
                    'month' => $month,
                ]
            ];
            $args['paged'] = 1;
        } else {
            $args['paged'] = $part;
        }

        $q = new WP_Query($args);

        $xml  = '<?xml version="1.0" encoding="' . esc_attr(get_bloginfo('charset')) . '"?>' . "\n";
        $xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";

        if ($q->have_posts()) {
            foreach ($q->posts as $post_id) {
                $loc = get_permalink($post_id);
                if (!$loc) continue;

                $lastmod = get_post_modified_time('c', true, $post_id);
                if (!$lastmod) {
                    $lastmod = get_post_time('c', true, $post_id);
                }

                $xml .= '<url>';
                $xml .= '<loc>' . esc_url($loc) . '</loc>';
                if ($lastmod) {
                    $xml .= '<lastmod>' . esc_html($lastmod) . '</lastmod>';
                }
                $xml .= '</url>' . "\n";
            }
        }

        $xml .= '</urlset>';
        return $xml;
    }

    // ==========================
    // RENDER: PAGES
    // ==========================

    private function render_pages() {
        $max_urls = (int) get_option(self::OPTION_PREFIX . 'max_urls', self::MAX_URLS_DEFAULT);

        $args = [
            'post_type'      => 'page',
            'post_status'    => 'publish',
            'fields'         => 'ids',
            'posts_per_page' => $max_urls,
            'orderby'        => 'date',
            'order'          => 'DESC',
            'no_found_rows'  => true,
        ];

        $q = new WP_Query($args);

        $xml  = '<?xml version="1.0" encoding="' . esc_attr(get_bloginfo('charset')) . '"?>' . "\n";
        $xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";

        if ($q->have_posts()) {
            foreach ($q->posts as $post_id) {
                $loc = get_permalink($post_id);
                if (!$loc) continue;

                $lastmod = get_post_modified_time('c', true, $post_id);
                if (!$lastmod) {
                    $lastmod = get_post_time('c', true, $post_id);
                }

                $xml .= '<url>';
                $xml .= '<loc>' . esc_url($loc) . '</loc>';
                if ($lastmod) {
                    $xml .= '<lastmod>' . esc_html($lastmod) . '</lastmod>';
                }
                $xml .= '</url>' . "\n";
            }
        }

        $xml .= '</urlset>';
        return $xml;
    }

    // ==========================
    // RENDER: TAXONOMIES
    // ==========================

    private function render_taxonomies() {
        $include_tax = get_option(self::OPTION_PREFIX . 'include_taxonomies', '1') === '1';
        if (!$include_tax) {
            return $this->empty_urlset();
        }

        $taxonomies = get_taxonomies([
            'public' => true,
        ], 'objects');

        $xml  = '<?xml version="1.0" encoding="' . esc_attr(get_bloginfo('charset')) . '"?>' . "\n";
        $xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";

        foreach ($taxonomies as $tax) {
            if (!$tax->public || !$tax->show_ui) continue;

            $terms = get_terms([
                'taxonomy'   => $tax->name,
                'hide_empty' => true,
                'fields'     => 'ids',
                'number'     => 5000,
            ]);

            if (empty($terms) || is_wp_error($terms)) {
                continue;
            }

            foreach ($terms as $term_id) {
                $loc = get_term_link((int) $term_id, $tax->name);
                if (is_wp_error($loc)) continue;

                $xml .= '<url>';
                $xml .= '<loc>' . esc_url($loc) . '</loc>';
                $xml .= '</url>' . "\n";
            }
        }

        $xml .= '</urlset>';
        return $xml;
    }

    // ==========================
    // RENDER: IMAGES
    // ==========================

    private function render_images($part = 1) {
        $include_images = get_option(self::OPTION_PREFIX . 'include_images', '1') === '1';
        if (!$include_images) {
            return $this->empty_urlset();
        }

        $max_urls = (int) get_option(self::OPTION_PREFIX . 'max_urls', self::MAX_URLS_DEFAULT);
        $part     = max(1, (int) $part);

        $args = [
            'post_type'      => 'attachment',
            'post_status'    => 'inherit',
            'post_mime_type' => 'image',
            'fields'         => 'ids',
            'posts_per_page' => $max_urls,
            'paged'          => $part,
            'orderby'        => 'date',
            'order'          => 'DESC',
            'no_found_rows'  => true,
        ];

        $q = new WP_Query($args);

        $xml  = '<?xml version="1.0" encoding="' . esc_attr(get_bloginfo('charset')) . '"?>' . "\n";
        $xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9" xmlns:image="http://www.google.com/schemas/sitemap-image/1.1">' . "\n";

        if ($q->have_posts()) {
            foreach ($q->posts as $att_id) {
                $src = wp_get_attachment_url($att_id);
                if (!$src) continue;

                $parent_id = wp_get_post_parent_id($att_id);
                $loc = $parent_id ? get_permalink($parent_id) : $src;

                $xml .= '<url>';
                $xml .= '<loc>' . esc_url($loc) . '</loc>';
                $xml .= '<image:image><image:loc>' . esc_url($src) . '</image:loc></image:image>';
                $xml .= '</url>' . "\n";
            }
        }

        $xml .= '</urlset>';
        return $xml;
    }

    // ==========================
    // HELPERS
    // ==========================

    private function empty_urlset() {
        $xml  = '<?xml version="1.0" encoding="' . esc_attr(get_bloginfo('charset')) . '"?>' . "\n";
        $xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9"></urlset>';
        return $xml;
    }

    private function has_pages() {
        $q = new WP_Query([
            'post_type'      => 'page',
            'post_status'    => 'publish',
            'posts_per_page' => 1,
            'fields'         => 'ids',
            'no_found_rows'  => true,
        ]);
        return $q->have_posts();
    }

    private function has_taxonomies() {
        $taxonomies = get_taxonomies([
            'public' => true,
        ], 'objects');

        foreach ($taxonomies as $tax) {
            $terms = get_terms([
                'taxonomy'   => $tax->name,
                'hide_empty' => true,
                'number'     => 1,
                'fields'     => 'ids',
            ]);
            if (!empty($terms) && !is_wp_error($terms)) {
                return true;
            }
        }
        return false;
    }

    private function has_attachments() {
        $q = new WP_Query([
            'post_type'      => 'attachment',
            'post_status'    => 'inherit',
            'post_mime_type' => 'image',
            'posts_per_page' => 1,
            'fields'         => 'ids',
            'no_found_rows'  => true,
        ]);
        return $q->have_posts();
    }

    private function get_post_counts_grouped($by_date = true) {
        global $wpdb;

        $result = [
            'total'   => 0,
            'by_date' => [],
        ];

        // Totale
        $total = (int) $wpdb->get_var("
            SELECT COUNT(ID)
            FROM {$wpdb->posts}
            WHERE post_type = 'post'
              AND post_status = 'publish'
        ");
        $result['total'] = $total;

        if ($by_date) {
            $rows = $wpdb->get_results("
                SELECT DATE_FORMAT(post_date_gmt, '%Y-%m') AS ym, COUNT(ID) AS c
                FROM {$wpdb->posts}
                WHERE post_type = 'post'
                  AND post_status = 'publish'
                GROUP BY ym
                ORDER BY ym DESC
            ");
            if (!empty($rows)) {
                foreach ($rows as $row) {
                    $result['by_date'][$row->ym] = (int) $row->c;
                }
            }
        }

        return $result;
    }

    private function get_image_counts() {
        global $wpdb;
        $count = (int) $wpdb->get_var("
            SELECT COUNT(ID)
            FROM {$wpdb->posts}
            WHERE post_type = 'attachment'
              AND post_mime_type LIKE 'image/%'
              AND post_status = 'inherit'
        ");
        return $count;
    }

    // ==========================
    // ADMIN
    // ==========================

    public function admin_menu() {
        add_options_page(
            'Lean Bunker Sitemap',
            'Lean Bunker Sitemap',
            'manage_options',
            'lean-bunker-sitemap',
            [$this, 'render_settings_page']
        );
    }

    public function render_settings_page() {
        if (!current_user_can('manage_options')) {
            return;
        }

        if (isset($_POST['lb_sitemap_save']) && check_admin_referer('lb_sitemap_save_settings')) {
            $max_urls = isset($_POST['max_urls']) ? max(100, (int) $_POST['max_urls']) : self::MAX_URLS_DEFAULT;
            update_option(self::OPTION_PREFIX . 'max_urls', $max_urls);
            update_option(self::OPTION_PREFIX . 'split_by_date', isset($_POST['split_by_date']) ? '1' : '0');
            update_option(self::OPTION_PREFIX . 'include_images', isset($_POST['include_images']) ? '1' : '0');
            update_option(self::OPTION_PREFIX . 'include_taxonomies', isset($_POST['include_taxonomies']) ? '1' : '0');
            echo '<div class="updated"><p>Impostazioni salvate.</p></div>';
        }

        $max_urls      = (int) get_option(self::OPTION_PREFIX . 'max_urls', self::MAX_URLS_DEFAULT);
        $split_by_date = get_option(self::OPTION_PREFIX . 'split_by_date', '1') === '1';
        $include_images = get_option(self::OPTION_PREFIX . 'include_images', '1') === '1';
        $include_tax   = get_option(self::OPTION_PREFIX . 'include_taxonomies', '1') === '1';

        $index_url = esc_url(trailingslashit(home_url('/')) . 'sitemap-index.xml');
        ?>
        <div class="wrap">
            <h1>Lean Bunker Sitemap</h1>
            <p>Sitemap XML universale, leggera e scalabile. Nessuna configurazione obbligatoria: funziona già così.</p>

            <h2>Sitemap Index</h2>
            <p><a href="<?php echo $index_url; ?>" target="_blank"><?php echo $index_url; ?></a></p>

            <form method="post">
                <?php wp_nonce_field('lb_sitemap_save_settings'); ?>

                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><label for="max_urls">URL massimi per sitemap</label></th>
                        <td>
                            <input type="number" name="max_urls" id="max_urls" value="<?php echo esc_attr($max_urls); ?>" min="100" step="100">
                            <p class="description">Default 2000. Per siti enormi puoi ridurre per alleggerire le query.</p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">Suddivisione per anno/mese</th>
                        <td>
                            <label>
                                <input type="checkbox" name="split_by_date" <?php checked($split_by_date); ?>>
                                Attiva suddivisione temporale automatica per i post (consigliato per siti grandi).
                            </label>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">Sitemap immagini</th>
                        <td>
                            <label>
                                <input type="checkbox" name="include_images" <?php checked($include_images); ?>>
                                Includi sitemap immagini (image sitemap). Utile per siti editoriali e media.
                            </label>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">Sitemap tassonomie</th>
                        <td>
                            <label>
                                <input type="checkbox" name="include_taxonomies" <?php checked($include_tax); ?>>
                                Includi categorie, tag e tassonomie pubbliche.
                            </label>
                        </td>
                    </tr>
                </table>

                <p class="submit">
                    <button type="submit" name="lb_sitemap_save" class="button button-primary">Salva impostazioni</button>
                </p>
            </form>
        </div>
        <?php
    }

    // ==========================
    // ROBOTS.TXT
    // ==========================

    public function add_sitemap_to_robots($output, $public) {
        if ('0' === $public) {
            return $output;
        }

        $index_url = trailingslashit(home_url('/')) . 'sitemap-index.xml';

        if (strpos($output, 'Sitemap:') === false) {
            $output .= "\nSitemap: " . esc_url($index_url) . "\n";
        }

        return $output;
    }
}

new Lean_Bunker_Sitemap();