<?php
/**
 * Plugin Name: AI Content Auditor
 * Description: Analizza i contenuti di articoli, pagine e custom post type. Mostra i risultati nella lista e in un metabox.
 * Version: 0.0.4
 * Author: Tu
 */

if (!defined('ABSPATH')) exit;

class AI_Content_Auditor {

    public function __construct() {

        // Colonne lista contenuti
        add_filter('manage_posts_columns', [$this, 'add_columns']);
        add_filter('manage_pages_columns', [$this, 'add_columns']);
        add_action('manage_posts_custom_column', [$this, 'render_columns'], 10, 2);
        add_action('manage_pages_custom_column', [$this, 'render_columns'], 10, 2);

        // Ordinamento colonne
        add_filter('manage_edit-post_sortable_columns', [$this, 'sortable_columns']);
        add_filter('manage_edit-page_sortable_columns', [$this, 'sortable_columns']);
        add_action('pre_get_posts', [$this, 'orderby_columns']);

        // Pulsante "Analizza tutti"
        add_action('restrict_manage_posts', [$this, 'add_scan_button']);
        add_action('wp_ajax_aca_run_scan', [$this, 'ajax_scan_batch']);

        // Metabox nell’editor
        add_action('add_meta_boxes', [$this, 'init_metabox']);
    }

    /* -------------------------
       COLONNE
    --------------------------*/

    public function add_columns($columns) {
        $columns['aca_quality'] = 'Quality';
        $columns['aca_words'] = 'Parole';
        $columns['aca_readability'] = 'Leggibilità';
        $columns['aca_alt'] = 'Img senza alt';
        $columns['aca_dup'] = 'Duplicati';
        return $columns;
    }

    public function render_columns($column, $post_id) {
        $analysis = get_post_meta($post_id, '_aca_analysis', true);

        if (!is_array($analysis)) {
            $analysis = [
                'word_count'      => '-',
                'readability'     => '-',
                'missing_alt'     => '-',
                'duplicate_score' => '-',
                'quality_score'   => '-',
            ];
        }

        switch ($column) {
            case 'aca_quality':
                echo '<strong>' . esc_html($analysis['quality_score']) . '</strong>';
                break;
            case 'aca_words':
                echo esc_html($analysis['word_count']);
                break;
            case 'aca_readability':
                echo esc_html($analysis['readability']);
                break;
            case 'aca_alt':
                echo esc_html($analysis['missing_alt']);
                break;
            case 'aca_dup':
                echo esc_html($analysis['duplicate_score']);
                break;
        }
    }

    /* -------------------------
       ORDINAMENTO
    --------------------------*/

    public function sortable_columns($columns) {
        $columns['aca_quality']     = 'aca_quality';
        $columns['aca_words']       = 'aca_words';
        $columns['aca_readability'] = 'aca_readability';
        $columns['aca_alt']         = 'aca_alt';
        $columns['aca_dup']         = 'aca_dup';
        return $columns;
    }

    public function orderby_columns($query) {
        if (!is_admin() || !$query->is_main_query()) return;

        $orderby = $query->get('orderby');

        $sortable = [
            'aca_quality',
            'aca_words',
            'aca_readability',
            'aca_alt',
            'aca_dup'
        ];

        if (in_array($orderby, $sortable)) {
            $query->set('meta_key', '_aca_analysis');
            $query->set('orderby', 'meta_value');
        }
    }

    /* -------------------------
       PULSANTE "ANALIZZA TUTTI"
    --------------------------*/

    public function add_scan_button() {
        $screen = get_current_screen();
        if (!post_type_supports($screen->post_type, 'editor')) return;

        echo '<button class="button aca-scan-all" type="button">Analizza tutti</button>';

        ?>
        <script>
        jQuery(function($){
            $('.aca-scan-all').on('click', function(){
                if(!confirm("Avviare l'analisi di tutti i contenuti?")) return;

                let offset = 0;
                let batch  = 500;

                function runBatch() {
                    $.post(ajaxurl, {
                        action: 'aca_run_scan',
                        offset: offset,
                        batch: batch,
                        post_type: '<?php echo esc_js($screen->post_type); ?>'
                    }, function(resp){
                        if(resp.done){
                            alert("Analisi completata");
                            location.reload();
                        } else {
                            offset += batch;
                            runBatch();
                        }
                    });
                }

                runBatch();
            });
        });
        </script>
        <?php
    }

    public function ajax_scan_batch() {
        global $wpdb;

        $offset    = intval($_POST['offset']);
        $batch     = intval($_POST['batch']);
        $post_type = sanitize_text_field($_POST['post_type']);

        $posts = $wpdb->get_results($wpdb->prepare("
            SELECT ID FROM $wpdb->posts
            WHERE post_type = %s AND post_status = 'publish'
            ORDER BY ID ASC
            LIMIT %d OFFSET %d
        ", $post_type, $batch, $offset));

        if (!$posts) {
            wp_send_json(['done' => true]);
        }

        foreach ($posts as $p) {
            $post = get_post($p->ID);
            $analysis = $this->analyze_post($post);
            update_post_meta($post->ID, '_aca_analysis', $analysis);
        }

        wp_send_json(['done' => false]);
    }

    /* -------------------------
       METABOX
    --------------------------*/

    public function init_metabox() {
        $post_types = get_post_types(['public' => true], 'names');

        foreach ($post_types as $pt) {
            if (!post_type_supports($pt, 'editor')) continue;

            add_meta_box(
                'aca_metabox',
                'AI Content Auditor',
                [$this, 'render_metabox'],
                $pt,
                'side',
                'high'
            );
        }
    }

    public function render_metabox($post) {
        $analysis = get_post_meta($post->ID, '_aca_analysis', true);

        echo '<style>
            .aca-box p { margin: 4px 0; }
            .aca-score { font-size: 22px; font-weight: bold; }
        </style>';

        if (!is_array($analysis)) {
            echo '<p>Nessuna analisi disponibile.</p>';
            return;
        }

        echo '<div class="aca-box">';
        echo '<p class="aca-score">Quality: ' . esc_html($analysis['quality_score']) . '</p>';
        echo '<p><strong>Parole:</strong> ' . esc_html($analysis['word_count']) . '</p>';
        echo '<p><strong>Leggibilità:</strong> ' . esc_html($analysis['readability']) . '</p>';
        echo '<p><strong>Img senza alt:</strong> ' . esc_html($analysis['missing_alt']) . '</p>';
        echo '<p><strong>Duplicati:</strong> ' . esc_html($analysis['duplicate_score']) . '</p>';
        echo '</div>';
    }

    /* -------------------------
       ANALISI CONTENUTI
    --------------------------*/

    public function analyze_post($post) {
        $content_raw  = $post->post_content;
        $content_text = wp_strip_all_tags($content_raw);

        $word_count      = str_word_count($this->normalize_text($content_text));
        $readability     = $this->flesch_score($content_text);
        $missing_alt     = $this->count_missing_alt($content_raw);
        $duplicate_score = $this->duplicate_check($content_text);

        $quality_score   = $this->quality_score($word_count, $readability, $missing_alt, $duplicate_score);

        return [
            'word_count'      => $word_count,
            'readability'     => $readability,
            'missing_alt'     => $missing_alt,
            'duplicate_score' => $duplicate_score,
            'quality_score'   => $quality_score,
        ];
    }

    private function normalize_text($text) {
        $text = preg_replace('/[^\p{L}\p{N}\s]/u', ' ', $text);
        $text = preg_replace('/\s+/', ' ', $text);
        return trim($text);
    }

    private function flesch_score($text) {
        $sentences = max(1, preg_match_all('/[.!?]+/', $text, $m));
        $words     = max(1, str_word_count($this->normalize_text($text)));
        $syllables = max(1, $this->estimate_syllables($text));

        $score = 206.835 - (1.015 * ($words / $sentences)) - (84.6 * ($syllables / $words));
        return max(0, min(100, round($score)));
    }

    private function estimate_syllables($text) {
        $text = strtolower($text);
        $text = preg_replace('/[^a-zàèéìòóù]/u', ' ', $text);
        $words = explode(' ', $text);
        $syllables = 0;

        foreach ($words as $word) {
            if (!$word) continue;
            $vowels = preg_match_all('/[aeiouàèéìòóù]/u', $word, $m);
            $syllables += max(1, $vowels);
        }

        return $syllables;
    }

    private function count_missing_alt($html) {
        $missing = 0;
        if (preg_match_all('/<img[^>]*>/i', $html, $matches)) {
            foreach ($matches[0] as $img) {
                if (!preg_match('/\salt\s*=\s*"[^\"]*"/i', $img)) {
                    $missing++;
                }
            }
        }
        return $missing;
    }

    private function duplicate_check($content) {
        global $wpdb;

        $content = $this->normalize_text($content);
        if (!$content) return 0;

        $words = explode(' ', $content);
        $snippet_words = array_slice($words, 0, 20);
        if (count($snippet_words) < 10) return 0;

        $snippet = implode(' ', $snippet_words);
        $like = '%' . $wpdb->esc_like($snippet) . '%';

        $count = $wpdb->get_var($wpdb->prepare("
            SELECT COUNT(ID) FROM $wpdb->posts
            WHERE post_status = 'publish'
              AND post_content LIKE %s
            LIMIT 50
        ", $like));

        if ($count <= 1) return 0;
        if ($count >= 5) return 100;

        return round(($count - 1) / 4 * 100);
    }

    private function quality_score($word_count, $readability, $missing_alt, $duplicate_score) {
        $score = 0;

        if ($word_count < 300) $score += 10;
        elseif ($word_count < 600) $score += 40;
        elseif ($word_count <= 2000) $score += 70;
        else $score += 50;

        $score += ($readability / 2);

        if ($missing_alt > 0) $score -= min(20, $missing_alt * 5);

        $score -= ($duplicate_score / 3);

        return max(0, min(100, round($score)));
    }
}

new AI_Content_Auditor();