<?php
/**
 * Plugin Name: Lean Bunker Stat
 * Description: Traccia visite umane (totali e visitatori unici) per post, pagine e media. Colonne ordinabili in admin, con filtro avanzato per data e aggregazioni per categoria/tag.
 * Version: 1.3.4
 * Author: Riccardo Bastillo
 * License: GPL-2.0+
 * Text Domain: lean-bunker-stat
 * Requires at least: 5.6
 * Requires PHP: 5.6
 */

defined('ABSPATH') or exit;

class Lean_Bunker_Stat {

    private static $instance = null;

    public static function init() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));

        if (is_multisite()) {
            add_action('wpmu_new_blog', array($this, 'activate_for_new_site'));
        }

        if (is_admin()) {
            add_action('admin_init', array($this, 'maybe_upgrade_schema'));
            add_action('admin_init', array($this, 'maybe_cleanup_old_data'));
            add_action('restrict_manage_posts', array($this, 'render_date_filter'));
            add_action('admin_footer', array($this, 'inject_global_visit_summary_in_pagination'));
            add_action('restrict_manage_tags', array($this, 'render_date_filter'));
            add_action('admin_footer-edit-tags.php', array($this, 'inject_global_visit_summary_in_pagination'));
            $this->setup_admin_columns();
            $this->setup_term_admin_columns();
        }

        if (!is_admin()) {
            add_action('template_redirect', array($this, 'track_view'), 0);
        }
    }

    // ─── OTTIENI NOME TABELLA DINAMICO (MULTISITE SAFE) ─────────────

    private function get_table_name() {
        global $wpdb;
        return $wpdb->prefix . 'bunker_stat_views';
    }

    // ─── ATTIVAZIONE ───────────────────────────────────────────────

    public function activate() {
        global $wpdb;
        $table_name = $this->get_table_name();
        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE {$table_name} (
            id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
            object_type VARCHAR(20) NOT NULL DEFAULT 'post',
            object_id BIGINT UNSIGNED NOT NULL,
            primary_term_id BIGINT UNSIGNED NULL DEFAULT NULL,
            ip_hash CHAR(64) NOT NULL,
            visited_at DATETIME NOT NULL DEFAULT '0000-00-00 00:00:00',
            PRIMARY KEY (id),
            KEY object_type_object_id (object_type, object_id),
            KEY primary_term_id (primary_term_id),
            KEY visited_at (visited_at),
            KEY object_type_object_id_visited_at (object_type, object_id, visited_at)
        ) $charset_collate;";

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        dbDelta($sql);
    }

    public function activate_for_new_site($blog_id) {
        if (!function_exists('is_plugin_active_for_network')) return;
        if (is_plugin_active_for_network(plugin_basename(__FILE__))) {
            switch_to_blog($blog_id);
            $this->activate();
            restore_current_blog();
        }
    }

    public function deactivate() {
        // Dati mantenuti dopo disattivazione
    }

    // ─── FIX 1: MIGRAZIONE CON INDICI AGGIORNATI + primary_term_id ─

    public function maybe_upgrade_schema() {
        global $wpdb;
        $table_name = $this->get_table_name();

        // Verifica se la colonna object_type esiste
        $column_exists = $wpdb->get_results("SHOW COLUMNS FROM {$table_name} LIKE 'object_type'");

        if (empty($column_exists)) {
            // ✅ Migrazione completa con indici
            
            // 1. Aggiungi colonna object_type
            $wpdb->query("ALTER TABLE {$table_name} ADD COLUMN object_type VARCHAR(20) NOT NULL DEFAULT 'post' AFTER id");
            
            // 2. Rinomina post_id → object_id
            $wpdb->query("ALTER TABLE {$table_name} CHANGE COLUMN post_id object_id BIGINT UNSIGNED NOT NULL");
            
            // 3. Rimuovi indici vecchi
            $indexes = $wpdb->get_results("SHOW INDEX FROM {$table_name}");
            $index_names = wp_list_pluck($indexes, 'Key_name');
            
            if (in_array('post_id', $index_names)) {
                $wpdb->query("ALTER TABLE {$table_name} DROP INDEX post_id");
            }
            
            if (in_array('post_id_visited_at', $index_names)) {
                $wpdb->query("ALTER TABLE {$table_name} DROP INDEX post_id_visited_at");
            }
            
            // 4. Aggiungi nuovi indici
            $wpdb->query("ALTER TABLE {$table_name} ADD INDEX object_type_object_id (object_type, object_id)");
            $wpdb->query("ALTER TABLE {$table_name} ADD INDEX object_type_object_id_visited_at (object_type, object_id, visited_at)");
        }

        // ✅ Aggiungi colonna primary_term_id se non esiste
        $primary_term_exists = $wpdb->get_results("SHOW COLUMNS FROM {$table_name} LIKE 'primary_term_id'");
        
        if (empty($primary_term_exists)) {
            $wpdb->query("ALTER TABLE {$table_name} ADD COLUMN primary_term_id BIGINT UNSIGNED NULL DEFAULT NULL AFTER object_id");
            $wpdb->query("ALTER TABLE {$table_name} ADD INDEX primary_term_id (primary_term_id)");
        }
    }

    public function maybe_cleanup_old_data() {
        // Estendibile in futuro
    }

    // ─── GARANTISCE CHE LA TABELLA ESISTA (CON LOCK) ───────────────

    private function ensure_table_exists() {
        global $wpdb;
        $table_name = $this->get_table_name();
        
        // ✅ Fix race condition con transient lock
        if (get_transient('lbs_ensure_table_lock')) {
            return;
        }
        
        set_transient('lbs_ensure_table_lock', 1, 30);
        
        if (!$wpdb->get_var("SHOW TABLES LIKE '{$table_name}'")) {
            $this->activate();
        }
        
        delete_transient('lbs_ensure_table_lock');
    }

    // ─── TRACCIAMENTO VISITA ───────────────────────────────────────

    public function track_view() {
        if (is_admin() || is_feed() || is_preview()) {
            return;
        }

        // ✅ Escludi utenti loggati con capacità di editing
        if (is_user_logged_in() && current_user_can('edit_posts')) {
            return;
        }

        $this->ensure_table_exists();

        $user_agent = isset($_SERVER['HTTP_USER_AGENT']) ? $_SERVER['HTTP_USER_AGENT'] : '';
        if (preg_match('/(bot|crawl|spider|slurp|teoma|ia_archiver|yandex|baidu|bingbot|googlebot|facebookexternalhit|facebookbot|whatsapp|linkedinbot|twitterbot)/i', $user_agent)) {
            return;
        }

        $ip = $this->get_real_ip();
        $ip_hash = hash_hmac('sha256', $ip, defined('AUTH_SALT') ? AUTH_SALT : 'lean-bunker-stat');
        $visited_at = current_time('mysql', true);

        global $wpdb;
        $table_name = $this->get_table_name();

        // ─── SOLO POST/PAGE/ATTACHMENT SINGOLO ──────────────────────

        if (is_singular()) {
            $post = get_queried_object();
            if (!$post || !isset($post->ID) || $post->post_status !== 'publish') {
                return;
            }

            $allowed_types = array('post', 'page', 'attachment');
            if (!in_array($post->post_type, $allowed_types, true)) {
                return;
            }

            $object_id = (int) $post->ID;
            $object_type = 'post';
            
            // ✅ Ottieni il termine primario (categoria/tag principale)
            $primary_term_id = $this->get_primary_term_id($post);

            $wpdb->query($wpdb->prepare(
                "INSERT INTO {$table_name} (object_type, object_id, primary_term_id, ip_hash, visited_at) VALUES (%s, %d, %d, %s, %s)",
                $object_type,
                $object_id,
                $primary_term_id,
                $ip_hash,
                $visited_at
            ));
        }
        
        // ❌ RIMOSSO: tracciamento archivi (categorie/tag)
        // Non c'è un posto naturale nell'admin per visualizzarli
        // e crea confusione per l'utente
    }

    // ─── OTTIENI IL TERMINE PRIMARIO DI UN POST ─────────────────────

    private function get_primary_term_id($post) {
        // Prova prima con le tassonomie gerarchiche (categorie)
        $categories = get_the_terms($post->ID, 'category');
        if (!empty($categories)) {
            // Restituisci il primo termine (o il primario se definito)
            $primary = reset($categories);
            return (int) $primary->term_id;
        }
        
        // Prova con tag
        $tags = get_the_terms($post->ID, 'post_tag');
        if (!empty($tags)) {
            $primary = reset($tags);
            return (int) $primary->term_id;
        }
        
        // Prova con altre tassonomie pubbliche
        $taxonomies = get_post_taxonomies($post->ID);
        foreach ($taxonomies as $taxonomy) {
            $terms = get_the_terms($post->ID, $taxonomy);
            if (!empty($terms)) {
                $primary = reset($terms);
                return (int) $primary->term_id;
            }
        }
        
        // Nessun termine trovato
        return null;
    }

    // ─── FIX: SICUREZZA IP (EVITA SPOOFING X-FORWARDED-FOR) ────────

    private function get_real_ip() {
        // ✅ Cloudflare: affidabile
        if (!empty($_SERVER['HTTP_CF_CONNECTING_IP']) && filter_var($_SERVER['HTTP_CF_CONNECTING_IP'], FILTER_VALIDATE_IP)) {
            return $_SERVER['HTTP_CF_CONNECTING_IP'];
        }
        
        // ✅ Proxy affidabili (nginx, varnish, ecc.)
        if (!empty($_SERVER['HTTP_X_REAL_IP']) && filter_var($_SERVER['HTTP_X_REAL_IP'], FILTER_VALIDATE_IP)) {
            return $_SERVER['HTTP_X_REAL_IP'];
        }
        
        // ✅ REMOTE_ADDR: sempre sicuro (non spoofabile)
        if (!empty($_SERVER['REMOTE_ADDR']) && filter_var($_SERVER['REMOTE_ADDR'], FILTER_VALIDATE_IP)) {
            return $_SERVER['REMOTE_ADDR'];
        }
        
        return '127.0.0.1';
    }

    // ─── RENDER FILTRO IN ADMIN (POST/PAGE) ─────────────────────────

    public function render_date_filter() {
        global $typenow;
        
        // ✅ INCLUDI attachment
        $allowed_types = array_merge(
            get_post_types(array('public' => true), 'names'),
            array('attachment')
        );
        
        if (!in_array($typenow, $allowed_types, true)) {
            return;
        }

        $start = isset($_GET['lbs_start']) ? sanitize_text_field($_GET['lbs_start']) : '';
        $end   = isset($_GET['lbs_end']) ? sanitize_text_field($_GET['lbs_end']) : '';

        echo '<style>
            label[for="m"], select[name="m"] { display: none !important; }
        </style>';

        echo '<label class="screen-reader-text" for="lbs_start">' . __('Filtro date', 'lean-bunker-stat') . '</label>';
        echo '<input type="date" name="lbs_start" id="lbs_start" value="' . esc_attr($start) . '" style="width:140px; margin-right:8px;" />';
        echo '<input type="date" name="lbs_end" id="lbs_end" value="' . esc_attr($end) . '" style="width:140px; margin-right:16px;" />';

        if ($start || $end) {
            $reset_url = remove_query_arg(array('lbs_start', 'lbs_end'));
            echo '<a href="' . esc_url($reset_url) . '" style="font-size:12px; color:#888; text-decoration:none; margin-left:8px;">&times; reset</a>';
        }
    }

    // ─── RIEPILOGO GLOBALE VT/VU NELLA PAGINAZIONE (POST/PAGE) ─────

    public function inject_global_visit_summary_in_pagination() {
        global $pagenow, $typenow;

        if ($pagenow !== 'edit.php' && $pagenow !== 'edit-tags.php') {
            return;
        }

        // ✅ Gestione sia per post che per termini
        if ($pagenow === 'edit.php') {
            $allowed_types = array_merge(
                get_post_types(array('public' => true), 'names'),
                array('attachment')
            );

            if (!in_array($typenow, $allowed_types, true)) {
                return;
            }
        }

        $this->ensure_table_exists();

        $totals = $this->get_global_visit_totals();

        echo '<style>
            #lean-bunker-stat-summary-pagination {
                font-size: 13px;
                color: #555;
                margin-right: 16px;
                display: inline-block;
                vertical-align: middle;
                font-family: monospace;
                font-weight: bold;
            }
            #lean-bunker-stat-summary-pagination span {
                margin: 0 8px;
            }
        </style>';

        echo '<script>
            document.addEventListener("DOMContentLoaded", function() {
                var summaryHtml = \'<span id="lean-bunker-stat-summary-pagination">VT: ' . number_format_i18n($totals['total']) . ' &nbsp; VU: ' . number_format_i18n($totals['unique']) . '</span>\';
                
                var paginationContainer = document.querySelector(".tablenav-pages");
                if (!paginationContainer) return;

                var displayingNum = paginationContainer.querySelector(".displaying-num");
                if (displayingNum) {
                    displayingNum.insertAdjacentHTML("beforebegin", summaryHtml);
                } else {
                    paginationContainer.insertAdjacentHTML("afterbegin", summaryHtml);
                }
            });
        </script>';
    }

    // ─── TOTALE COMPLESSO DI TUTTI I POST FILTRATI ──────────────────

    private function get_filtered_post_ids() {
        global $wp_query;
        
        // ✅ Ottieni TUTTI gli ID dei post filtrati (non solo pagina corrente)
        $query_args = array(
            'fields' => 'ids',
            'posts_per_page' => -1, // Tutti i post
            'post_status' => 'publish',
            'no_found_rows' => true,
            'update_post_meta_cache' => false,
            'update_post_term_cache' => false
        );
        
        // Mantieni tutti i filtri attivi dalla query principale
        $filters = array('author', 'author_name', 'author__in', 'author__not_in', 'cat', 'category_name', 'category__and', 'category__in', 'category__not_in', 'tag', 'tag_id', 'tag__and', 'tag__in', 'tag__not_in', 'tag_slug__and', 'tag_slug__in', 'tax_query', 's', 'year', 'monthnum', 'day', 'post_type');
        
        foreach ($filters as $filter) {
            if (isset($wp_query->query_vars[$filter])) {
                $query_args[$filter] = $wp_query->query_vars[$filter];
            }
        }
        
        $filtered_posts = new WP_Query($query_args);
        return !empty($filtered_posts->posts) ? $filtered_posts->posts : array();
    }

    // ─── OTTIENI TUTTI GLI ID DEI TERMINI FILTRATI ──────────────────

    private function get_filtered_term_ids() {
        global $wp_query;
        
        // Per edit-tags.php, i termini sono già caricati
        global $terms;
        
        if (!empty($terms)) {
            return wp_list_pluck($terms, 'term_id');
        }
        
        return array();
    }

    // ─── CALCOLO TOTALE CON TUTTI I FILTRI ──────────────────────────

    private function get_global_visit_totals() {
        global $wpdb;
        $table_name = $this->get_table_name();

        if (!$wpdb->get_var("SHOW TABLES LIKE '{$table_name}'")) {
            return array('total' => 0, 'unique' => 0);
        }

        global $pagenow;
        
        // ─── POST/PAGE ──────────────────────────────────────────────
        
        if ($pagenow === 'edit.php') {
            $post_ids = $this->get_filtered_post_ids();

            if (empty($post_ids)) {
                return array('total' => 0, 'unique' => 0);
            }

            $placeholders = str_repeat('%d,', count($post_ids) - 1) . '%d';
            $where = "object_type = 'post' AND object_id IN (" . $placeholders . ")";
            $params = $post_ids;

            if ($this->is_period_mode()) {
                $start = isset($_GET['lbs_start']) ? sanitize_text_field($_GET['lbs_start']) : '';
                $end   = isset($_GET['lbs_end']) ? sanitize_text_field($_GET['lbs_end']) : '';

                if ($start && !preg_match('/^\d{4}-\d{2}-\d{2}$/', $start)) $start = '';
                if ($end   && !preg_match('/^\d{4}-\d{2}-\d{2}$/', $end))   $end   = '';

                if ($start) {
                    $where .= " AND visited_at >= %s";
                    $params[] = $start . ' 00:00:00';
                }
                if ($end) {
                    $where .= " AND visited_at <= %s";
                    $params[] = $end . ' 23:59:59';
                }
            }

            $subquery = $wpdb->prepare(
                "SELECT COUNT(*) AS total,
                        COUNT(DISTINCT ip_hash) AS `unique`
                 FROM {$table_name}
                 WHERE {$where}",
                $params
            );

            $row = $wpdb->get_row($subquery, ARRAY_A);
            return $row ? $row : array('total' => 0, 'unique' => 0);
        }

        // ─── TERMINI ────────────────────────────────────────────────
        
        if ($pagenow === 'edit-tags.php') {
            $term_ids = $this->get_filtered_term_ids();

            if (empty($term_ids)) {
                return array('total' => 0, 'unique' => 0);
            }

            // ✅ Usa primary_term_id per un calcolo veloce e preciso
            $placeholders = str_repeat('%d,', count($term_ids) - 1) . '%d';
            $where = "object_type = 'post' AND primary_term_id IN (" . $placeholders . ")";
            $params = $term_ids;

            if ($this->is_period_mode()) {
                $start = isset($_GET['lbs_start']) ? sanitize_text_field($_GET['lbs_start']) : '';
                $end   = isset($_GET['lbs_end']) ? sanitize_text_field($_GET['lbs_end']) : '';

                if ($start && !preg_match('/^\d{4}-\d{2}-\d{2}$/', $start)) $start = '';
                if ($end   && !preg_match('/^\d{4}-\d{2}-\d{2}$/', $end))   $end   = '';

                if ($start) {
                    $where .= " AND visited_at >= %s";
                    $params[] = $start . ' 00:00:00';
                }
                if ($end) {
                    $where .= " AND visited_at <= %s";
                    $params[] = $end . ' 23:59:59';
                }
            }

            $subquery = $wpdb->prepare(
                "SELECT COUNT(*) AS total,
                        COUNT(DISTINCT ip_hash) AS `unique`
                 FROM {$table_name}
                 WHERE {$where}",
                $params
            );

            $row = $wpdb->get_row($subquery, ARRAY_A);
            return $row ? $row : array('total' => 0, 'unique' => 0);
        }

        return array('total' => 0, 'unique' => 0);
    }

    // ─── COLONNE ADMIN PER POST/PAGE ────────────────────────────────

    private function setup_admin_columns() {
        // ✅ INCLUDI attachment
        $post_types = array_merge(
            get_post_types(array('public' => true), 'names'),
            array('attachment')
        );
        
        foreach ($post_types as $type) {
            add_filter("manage_{$type}_posts_columns", array($this, 'add_columns'), 20);
            add_action("manage_{$type}_posts_custom_column", array($this, 'render_columns'), 10, 2);
            add_filter("manage_edit-{$type}_sortable_columns", array($this, 'sortable_columns'));
        }
        add_action('pre_get_posts', array($this, 'handle_orderby'));
    }

    public function add_columns($columns) {
        $columns['lbs_vu'] = '<abbr title="Visitatori Unici Umani">VU</abbr>';
        $columns['lbs_vt'] = '<abbr title="Visite Totali Umane">VT</abbr>';
        return $columns;
    }

    public function render_columns($column, $post_id) {
        if (!in_array($column, array('lbs_vu', 'lbs_vt'))) return;

        $this->ensure_table_exists();
        $counts = $this->get_visit_counts('post', $post_id);
        $value = ($column === 'lbs_vu') ? $counts['unique'] : $counts['total'];

        echo '<span style="font-family: monospace;">' . number_format_i18n($value) . '</span>';
    }

    public function sortable_columns($columns) {
        $columns['lbs_vu'] = 'lbs_vu';
        $columns['lbs_vt'] = 'lbs_vt';
        return $columns;
    }

    private function is_period_mode() {
        $start = isset($_GET['lbs_start']) ? sanitize_text_field($_GET['lbs_start']) : '';
        $end   = isset($_GET['lbs_end']) ? sanitize_text_field($_GET['lbs_end']) : '';
        return !empty($start) || !empty($end);
    }

    private function get_visit_counts($object_type, $object_id) {
        global $wpdb;
        $table_name = $this->get_table_name();

        if (!$wpdb->get_var("SHOW TABLES LIKE '{$table_name}'")) {
            return array('total' => 0, 'unique' => 0);
        }

        if ($this->is_period_mode()) {
            $start = isset($_GET['lbs_start']) ? sanitize_text_field($_GET['lbs_start']) : '';
            $end   = isset($_GET['lbs_end']) ? sanitize_text_field($_GET['lbs_end']) : '';

            if ($start && !preg_match('/^\d{4}-\d{2}-\d{2}$/', $start)) $start = '';
            if ($end   && !preg_match('/^\d{4}-\d{2}-\d{2}$/', $end))   $end   = '';

            $date_condition = '';
            $params = array($object_type, $object_id);
            if ($start) {
                $date_condition .= " AND visited_at >= %s";
                $params[] = $start . ' 00:00:00';
            }
            if ($end) {
                $date_condition .= " AND visited_at <= %s";
                $params[] = $end . ' 23:59:59';
            }

            $query = $wpdb->prepare(
                "SELECT COUNT(*) AS total,
                        COUNT(DISTINCT ip_hash) AS `unique`
                 FROM {$table_name}
                 WHERE object_type = %s AND object_id = %d {$date_condition}",
                $params
            );
        } else {
            $query = $wpdb->prepare(
                "SELECT COUNT(*) AS total,
                        COUNT(DISTINCT ip_hash) AS `unique`
                 FROM {$table_name}
                 WHERE object_type = %s AND object_id = %d",
                $object_type,
                $object_id
            );
        }

        $row = $wpdb->get_row($query, ARRAY_A);
        return $row ? $row : array('total' => 0, 'unique' => 0);
    }

    public function handle_orderby($query) {
        if (!is_admin() || !$query->is_main_query()) return;

        $orderby = $query->get('orderby');
        if (!in_array($orderby, array('lbs_vu', 'lbs_vt'))) return;

        global $wpdb;
        $table_name = $this->get_table_name();

        if (!$wpdb->get_var("SHOW TABLES LIKE '{$table_name}'")) {
            return;
        }

        $query->set('orderby', 'none');

        if ($this->is_period_mode()) {
            $start = isset($_GET['lbs_start']) ? sanitize_text_field($_GET['lbs_start']) : '';
            $end   = isset($_GET['lbs_end']) ? sanitize_text_field($_GET['lbs_end']) : '';

            if ($start && !preg_match('/^\d{4}-\d{2}-\d{2}$/', $start)) $start = '';
            if ($end   && !preg_match('/^\d{4}-\d{2}-\d{2}$/', $end))   $end   = '';

            $date_condition = '';
            $params = array();
            if ($start) {
                $date_condition .= " AND visited_at >= %s";
                $params[] = $start . ' 00:00:00';
            }
            if ($end) {
                $date_condition .= " AND visited_at <= %s";
                $params[] = $end . ' 23:59:59';
            }

            $subquery = $wpdb->prepare(
                "SELECT object_id AS post_id,
                        COUNT(*) AS human_views,
                        COUNT(DISTINCT ip_hash) AS `unique_visitors`
                 FROM {$table_name}
                 WHERE object_type = 'post' AND 1=1 {$date_condition}
                 GROUP BY object_id",
                $params
            );
        } else {
            $subquery = "SELECT object_id AS post_id,
                                COUNT(*) AS human_views,
                                COUNT(DISTINCT ip_hash) AS `unique_visitors`
                         FROM {$table_name}
                         WHERE object_type = 'post'
                         GROUP BY object_id";
        }

        // ✅ Usa closure invece di funzioni globali persistenti
        $join_callback = function($join) use ($subquery, $wpdb) {
            return $join . " LEFT JOIN ({$subquery}) AS lbs ON lbs.post_id = {$wpdb->posts}.ID ";
        };
        
        $orderby_field = ($orderby === 'lbs_vu') ? 'unique_visitors' : 'human_views';
        $orderby_callback = function($orderby_sql) use ($orderby_field) {
            return "lbs.{$orderby_field} DESC";
        };

        // ✅ Aggiungi i filtri
        add_filter('posts_join', $join_callback, 10, 1);
        add_filter('posts_orderby', $orderby_callback, 10, 1);
        
        // ✅ Rimuovili automaticamente dopo l'uso
        add_filter('the_posts', function($posts) use ($join_callback, $orderby_callback) {
            remove_filter('posts_join', $join_callback);
            remove_filter('posts_orderby', $orderby_callback);
            remove_filter('the_posts', current_filter());
            return $posts;
        }, 10, 1);
    }

    // ─── COLONNE ADMIN PER TERMINI (CATEGORIE/TAG) ──────────────────

    private function setup_term_admin_columns() {
        $taxonomies = get_taxonomies(array('public' => true), 'names');
        
        foreach ($taxonomies as $taxonomy) {
            add_filter("manage_edit-{$taxonomy}_columns", array($this, 'add_term_columns'), 20);
            add_filter("manage_{$taxonomy}_custom_column", array($this, 'render_term_columns'), 10, 3);
            add_filter("manage_edit-{$taxonomy}_sortable_columns", array($this, 'sortable_term_columns'));
        }
        add_filter('terms_clauses', array($this, 'handle_term_orderby'), 10, 3);
    }

    public function add_term_columns($columns) {
        $columns['lbs_vu'] = '<abbr title="Visitatori Unici Umani">VU</abbr>';
        $columns['lbs_vt'] = '<abbr title="Visite Totali Umane">VT</abbr>';
        return $columns;
    }

    public function render_term_columns($content, $column_name, $term_id) {
        if (!in_array($column_name, array('lbs_vu', 'lbs_vt'))) {
            return $content;
        }

        $this->ensure_table_exists();
        $counts = $this->get_term_visit_counts($term_id);
        $value = ($column_name === 'lbs_vu') ? $counts['unique'] : $counts['total'];

        return '<span style="font-family: monospace;">' . number_format_i18n($value) . '</span>';
    }

    public function sortable_term_columns($columns) {
        $columns['lbs_vu'] = 'lbs_vu';
        $columns['lbs_vt'] = 'lbs_vt';
        return $columns;
    }

    // ─── OTTIENI VISITE PER CATEGORIA/TAG (USANDO primary_term_id) ─

    private function get_term_visit_counts($term_id) {
        global $wpdb;
        $table_name = $this->get_table_name();

        if (!$wpdb->get_var("SHOW TABLES LIKE '{$table_name}'")) {
            return array('total' => 0, 'unique' => 0);
        }

        if ($this->is_period_mode()) {
            $start = isset($_GET['lbs_start']) ? sanitize_text_field($_GET['lbs_start']) : '';
            $end = isset($_GET['lbs_end']) ? sanitize_text_field($_GET['lbs_end']) : '';

            if ($start && !preg_match('/^\d{4}-\d{2}-\d{2}$/', $start)) $start = '';
            if ($end && !preg_match('/^\d{4}-\d{2}-\d{2}$/', $end)) $end = '';

            $date_condition = '';
            $params = array($term_id);
            if ($start) {
                $date_condition .= " AND visited_at >= %s";
                $params[] = $start . ' 00:00:00';
            }
            if ($end) {
                $date_condition .= " AND visited_at <= %s";
                $params[] = $end . ' 23:59:59';
            }

            $query = $wpdb->prepare(
                "SELECT COUNT(*) AS total,
                        COUNT(DISTINCT ip_hash) AS `unique`
                 FROM {$table_name}
                 WHERE object_type = 'post' AND primary_term_id = %d {$date_condition}",
                $params
            );
        } else {
            $query = $wpdb->prepare(
                "SELECT COUNT(*) AS total,
                        COUNT(DISTINCT ip_hash) AS `unique`
                 FROM {$table_name}
                 WHERE object_type = 'post' AND primary_term_id = %d",
                $term_id
            );
        }

        $row = $wpdb->get_row($query, ARRAY_A);
        return $row ? $row : array('total' => 0, 'unique' => 0);
    }

    public function handle_term_orderby($clauses, $taxonomies, $args) {
        global $pagenow;

        if ($pagenow !== 'edit-tags.php') {
            return $clauses;
        }

        if (!isset($_GET['orderby']) || !in_array($_GET['orderby'], array('lbs_vu', 'lbs_vt'))) {
            return $clauses;
        }

        global $wpdb;
        $table_name = $this->get_table_name();
        $orderby = sanitize_text_field($_GET['orderby']);
        $order = isset($_GET['order']) && $_GET['order'] === 'asc' ? 'ASC' : 'DESC';
        
        $date_condition = '';
        $params = array();
        
        if ($this->is_period_mode()) {
            $start = isset($_GET['lbs_start']) ? sanitize_text_field($_GET['lbs_start']) : '';
            $end = isset($_GET['lbs_end']) ? sanitize_text_field($_GET['lbs_end']) : '';
            
            if ($start && preg_match('/^\d{4}-\d{2}-\d{2}$/', $start)) {
                $date_condition .= " AND visited_at >= %s";
                $params[] = $start . ' 00:00:00';
            }
            if ($end && preg_match('/^\d{4}-\d{2}-\d{2}$/', $end)) {
                $date_condition .= " AND visited_at <= %s";
                $params[] = $end . ' 23:59:59';
            }
        }
        
        $field = ($orderby === 'lbs_vu') ? 'unique_visitors' : 'total';
        
        // ✅ Ordinamento veloce usando primary_term_id
        $subquery = $wpdb->prepare(
            "SELECT
                primary_term_id AS term_id,
                COUNT(*) AS total,
                COUNT(DISTINCT ip_hash) AS unique_visitors
             FROM {$table_name}
             WHERE object_type = 'post' AND primary_term_id IS NOT NULL {$date_condition}
             GROUP BY primary_term_id",
            $params
        );
        
        $clauses['join'] .= " LEFT JOIN ({$subquery}) AS lbs ON lbs.term_id = {$wpdb->term_taxonomy}.term_id ";
        $clauses['orderby'] = "lbs.{$field} {$order}";
        
        return $clauses;
    }
}

add_action('plugins_loaded', function() {
    Lean_Bunker_Stat::init();
});