<?php
/**
* Plugin Name: Lean Bunker SEO
* Description: Genera meta title, meta description, Open Graph, Schema.org e breadcrumb con logica nativa WordPress. Invia sitemap a Bing, Yandex, Baidu. Ping Google "di cortesia". Zero dipendenze esterne. Tutto in un file.
* Version: 2.2.2
* Author: Riccardo Bastillo
* Author URI: https://leanbunker.com
* License: GPL-2.0+
* License URI: https://www.gnu.org/licenses/gpl-2.0.html
* Text Domain: lean-bunker-seo
* Domain Path: /languages
* Requires at least: 6.0
* Tested up to: 6.7
* Requires PHP: 7.4
*
* Changelog 2.2.2:
* 🔔 NUOVO: Ping Google "di cortesia" (non affidabile al 100%, ma utile per completezza)
* 📊 NUOVO: Dashboard ping con stato attuale e cronologia ultimi ping
* 🎯 NUOVO: Bottone "Ping ora" per triggerare manualmente l'invio
* ✅ FIX: Verifica sitemap esistente prima di pingare (evita errori)
* 📝 FIX: Logging migliorato con timestamp, HTTP code e messaggi di errore
* 🛡️ FIX: Validazione URL sitemap prima dell'invio
* 📈 NUOVO: Contatore ping totali per ogni motore di ricerca
* 🎨 NUOVO: UI migliorata con icone stato e colori semantici
*
* Changelog 2.2.1:
* 🔒 FIX: Aggiunto nonce su save_meta() per sicurezza CSRF
* 🛡️ FIX: Non sovrascrive meta tags manuali durante bulk generation
* ⚡ FIX: Lock transiente per evitare race condition nel bulk
* 📊 FIX: Logging ping sitemap per debug futuro
* 🛡️ FIX: Fail-safe template title (fallback se token errati)
* ✅ Aggiunto bulk generation per tassonomie (categorie, tag, custom taxonomies)
* ✅ Migliorata robustezza per siti grandi (220.000+ articoli)
*/
if (!defined('ABSPATH')) exit;

// ============================================================================
// COSTANTI GLOBALI
// ============================================================================

// Whitelist schemi autorizzati per Schema.org (sicurezza + governance)
const LBS_ALLOWED_SCHEMA_TYPES = [
    'WebSite',
    'WebPage',
    'CollectionPage',
    'Article',
    'Person',
    'Organization',
    'ImageObject',
    'BreadcrumbList'
];

// Endpoint ping motori di ricerca
const LBS_PING_ENDPOINTS = [
    'google' => 'https://www.google.com/ping?sitemap=',
    'bing'   => 'https://www.bing.com/ping?sitemap=',
    'yandex' => 'https://webmaster.yandex.com/ping?sitemap=',
];

// ============================================================================
// CLASSE PRINCIPALE
// ============================================================================

class Lean_Bunker_SEO {
    private $option_prefix = 'lb_seo_';

    // ============================================================================
    // HOOKS DI INIZIALIZZAZIONE
    // ============================================================================

    public function __construct() {
        // Init & Core
        add_action('init', [$this, 'init']);

        // Admin UI
        add_action('admin_menu', [$this, 'admin_menu']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_scripts']);
        add_action('add_meta_boxes', [$this, 'add_metabox']);
        add_action('category_edit_form_fields', [$this, 'taxonomy_metabox_html'], 10, 2);
        add_action('post_tag_edit_form_fields', [$this, 'taxonomy_metabox_html'], 10, 2);
        add_action('edited_category', [$this, 'save_taxonomy_meta']);
        add_action('edited_post_tag', [$this, 'save_taxonomy_meta']);

        // Post Meta
        add_action('save_post', [$this, 'save_meta'], 10, 2);
        add_shortcode('lb_seo_description', [$this, 'shortcode_seo_desc']);
        add_shortcode('lb_breadcrumb', [$this, 'shortcode_breadcrumb']);

        // Auto-generation
        add_action('transition_post_status', [$this, 'auto_generate_on_publish'], 10, 3);

        // AJAX
        add_action('wp_ajax_lb_seo_generate', [$this, 'ajax_generate']);
        add_action('wp_ajax_lb_seo_get_missing', [$this, 'ajax_get_missing_posts']);
        add_action('wp_ajax_lb_seo_generate_single', [$this, 'ajax_generate_single']);
        add_action('wp_ajax_lb_seo_get_missing_taxonomies', [$this, 'ajax_get_missing_taxonomies']);
        add_action('wp_ajax_lb_seo_generate_taxonomy_single', [$this, 'ajax_generate_taxonomy_single']);
        add_action('wp_ajax_lb_seo_manual_ping', [$this, 'ajax_manual_ping']);

        // Frontend Output
        add_filter('pre_get_document_title', [$this, 'get_seo_title'], 20);
        add_action('wp_head', [$this, 'output_seo_meta'], 1);
        add_action('wp_head', [$this, 'output_noindex_nofollow'], 5);
        add_action('wp_footer', [$this, 'output_schema_markup'], 1);
        add_filter('the_content', [$this, 'add_nofollow_external_links']);

        // Sitemap & Ping
        add_filter('robots_txt', [$this, 'add_sitemap_to_robots'], 10, 2);
        add_action('plugins_loaded', [$this, 'schedule_pinger']);
    }

    public function init() {
        // Nessuna API key richiesta - tutto nativo WordPress
    }

    // ============================================================================
    // SEO: META TAGS & OPEN GRAPH
    // ============================================================================

    /**
    * Genera il meta title SEO con template e fail-safe
    */
    public function get_seo_title($title) {
        if (!is_singular()) return $title;

        $post_id = get_queried_object_id();
        if (!$post_id) return $title;

        $seo_title = get_post_meta($post_id, '_lb_seo_title', true);
        if ($seo_title) {
            return $seo_title;
        }

        // Usa template globale
        $template = get_option($this->option_prefix . 'title_template', '{title} - {sitename}');
        $post_title = wp_strip_all_tags(get_the_title($post_id));
        $site_name = get_bloginfo('name');

        // Token {sep} = separatore intelligente
        $sep = apply_filters('document_title_separator', '-');

        // Token {category} = categoria principale (solo per post)
        $category = '';
        if (get_post_type($post_id) === 'post') {
            $categories = get_the_category($post_id);
            if (!empty($categories)) {
                $category = $categories[0]->name;
            }
        }

        // Sostituisci token
        $template = str_replace('{title}', $post_title, $template);
        $template = str_replace('{sitename}', $site_name, $template);
        $template = str_replace('{category}', $category, $template);
        $template = str_replace('{sep}', $sep, $template);

        // FAIL-SAFE: se ci sono ancora token non sostituiti, usa fallback
        if (strpos($template, '{') !== false) {
            $template = $post_title . ' - ' . $site_name;
        }

        // Trim e limita a 60 caratteri
        $template = trim($template);
        if (mb_strlen($template) > 60) {
            $template = mb_substr($template, 0, 57) . '...';
        }

        return $template;
    }

    /**
    * Output meta tags e Open Graph in <head>
    */
    public function output_seo_meta() {
        // SEO per archivi tassonomie
        if (is_category() || is_tag() || is_tax()) {
            $this->output_taxonomy_seo();
            return;
        }

        // SEO per pagine singolari
        if (!is_singular()) return;

        $post_id = get_queried_object_id();
        if (!$post_id) return;

        $title = get_post_meta($post_id, '_lb_seo_title', true) ?: wp_strip_all_tags(get_the_title($post_id));
        $desc  = get_post_meta($post_id, '_lb_seo_desc', true);

        if (!$desc) {
            $desc = get_the_excerpt($post_id);
            if (!$desc) {
                $post = get_post($post_id);
                $desc = wp_trim_words(wp_strip_all_tags($post->post_content ?? ''), 25, '...');
            }
        }

        $url = get_permalink($post_id);
        $image = get_the_post_thumbnail_url($post_id, 'large')
            ?: get_site_icon_url(512)
            ?: get_option($this->option_prefix . 'og_fallback_image');

        // Determina il tipo Open Graph: article per post, website per pagine/CPT
        $post = get_post($post_id);
        $og_type = ($post && $post->post_type === 'post') ? 'article' : 'website';

        // Meta Description
        if ($desc) {
            echo '<meta name="description" content="' . esc_attr(wp_strip_all_tags($desc)) . '">' . "\n";
        }

        // Meta Robots per singolo post
        $robots = get_post_meta($post_id, '_lb_seo_robots', true);
        if ($robots) {
            echo '<meta name="robots" content="' . esc_attr($robots) . '">' . "\n";
        }

        // Canonical URL (essenziale per SEO)
        if ($url) {
            echo '<link rel="canonical" href="' . esc_url($url) . '">' . "\n";
        }

        // Open Graph Tags
        echo '<meta property="og:title" content="' . esc_attr($title) . '">' . "\n";
        echo '<meta property="og:description" content="' . esc_attr(wp_strip_all_tags($desc)) . '">' . "\n";
        echo '<meta property="og:url" content="' . esc_url($url) . '">' . "\n";
        echo '<meta property="og:type" content="' . esc_attr($og_type) . '">' . "\n";

        if ($image) {
            echo '<meta property="og:image" content="' . esc_url($image) . '">' . "\n";
            // Dimensioni consigliate per social (Facebook, LinkedIn, Discord)
            echo '<meta property="og:image:width" content="1200">' . "\n";
            echo '<meta property="og:image:height" content="630">' . "\n";
        }
    }

    /**
    * Output SEO per archivi tassonomie (categorie, tag, custom tax)
    */
    private function output_taxonomy_seo() {
        $term = get_queried_object();
        if (!$term || is_wp_error($term)) return;

        $title = get_term_meta($term->term_id, '_lb_seo_title', true);
        $desc = get_term_meta($term->term_id, '_lb_seo_desc', true);
        $robots = get_term_meta($term->term_id, '_lb_seo_robots', true);

        // Fallback se non impostato
        if (!$title) {
            $title = $term->name . ' - ' . get_bloginfo('name');
        }

        if (!$desc && !empty($term->description)) {
            $desc = wp_trim_words(wp_strip_all_tags($term->description), 25, '...');
        }

        $url = get_term_link($term);
        if (is_wp_error($url)) return;

        // Meta description
        if ($desc) {
            echo '<meta name="description" content="' . esc_attr(wp_strip_all_tags($desc)) . '">' . "\n";
        }

        // Meta robots
        if ($robots) {
            echo '<meta name="robots" content="' . esc_attr($robots) . '">' . "\n";
        }

        // Canonical
        echo '<link rel="canonical" href="' . esc_url($url) . '">' . "\n";

        // Open Graph per tassonomie
        echo '<meta property="og:title" content="' . esc_attr($title) . '">' . "\n";
        if ($desc) {
            echo '<meta property="og:description" content="' . esc_attr(wp_strip_all_tags($desc)) . '">' . "\n";
        }
        echo '<meta property="og:url" content="' . esc_url($url) . '">' . "\n";
        echo '<meta property="og:type" content="website">' . "\n";
    }

    /**
    * Output noindex/nofollow per archivi e ricerca (opzionale)
    */
    public function output_noindex_nofollow() {
        if (is_search() && get_option($this->option_prefix . 'noindex_search') === '1') {
            echo '<meta name="robots" content="noindex,follow">' . "\n";
            return;
        }

        if ((is_category() || is_tag() || is_author() || is_date()) && get_option($this->option_prefix . 'noindex_archives') === '1') {
            echo '<meta name="robots" content="noindex,follow">' . "\n";
        }
    }

    /**
    * Aggiunge rel="nofollow" ai link esterni (esclusi mailto: e tel:)
    */
    public function add_nofollow_external_links($content) {
        if (get_option($this->option_prefix . 'nofollow_external') !== '1') {
            return $content;
        }

        $home_host = parse_url(home_url(), PHP_URL_HOST);
        if (!$home_host) return $content;

        $dom = new DOMDocument();

        // Wrapper robusto per evitare problemi con HTML sporco
        libxml_use_internal_errors(true);
        $dom->loadHTML(
            '<html><body>' . mb_convert_encoding($content, 'HTML-ENTITIES', 'UTF-8') . '</body></html>',
            LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD
        );
        libxml_clear_errors();

        $links = $dom->getElementsByTagName('a');
        $modified = false;

        foreach ($links as $link) {
            $href = $link->getAttribute('href');
            if (!$href) continue;

            // Escludi mailto: e tel:
            $scheme = parse_url($href, PHP_URL_SCHEME);
            if (in_array($scheme, ['mailto', 'tel'], true)) {
                continue;
            }

            $link_host = parse_url($href, PHP_URL_HOST);
            if ($link_host && $link_host !== $home_host) {
                $existing_rel = $link->getAttribute('rel');
                $new_rel = trim($existing_rel . ' nofollow');
                $link->setAttribute('rel', $new_rel);
                $modified = true;
            }
        }

        if ($modified) {
            // Estrai solo il contenuto del body
            $body = $dom->getElementsByTagName('body')->item(0);
            if ($body) {
                $new_content = '';
                foreach ($body->childNodes as $child) {
                    $new_content .= $dom->saveHTML($child);
                }
                return $new_content;
            }
        }

        return $content;
    }

    // ============================================================================
    // SCHEMA.ORG JSON-LD
    // ============================================================================

    /**
    * Output Schema.org JSON-LD in <footer>
    * Posizionato in footer per evitare conflitti con altri plugin SEO
    */
    public function output_schema_markup() {
        // Schema per archivi tassonomie
        if (is_category() || is_tag() || is_tax()) {
            $term = get_queried_object();
            if ($term && !is_wp_error($term)) {
                $this->output_taxonomy_schema($term);
            }
            return;
        }

        // Schema per pagine singolari
        if (!is_singular()) {
            return;
        }

        global $post;

        // Protezione: post deve esistere e essere pubblicato
        if (!$post || !($post instanceof WP_Post) || $post->post_status !== 'publish') {
            return;
        }

        $schema_json = $this->generate_complete_schema($post->ID);
        if ($schema_json) {
            echo $schema_json;
        }
    }

    /**
    * Genera lo schema completo con fallback garantiti
    * Minimo 3 schemi core sempre presenti (WebSite + WebPage + Organization)
    */
    private function generate_complete_schema($post_id) {
        $schemas = [];

        // === CORE: SEMPRE presenti (garantiti) ===

        // WebSite (sempre)
        $website = $this->generate_website_schema();
        if ($website) {
            $schemas[] = $website;
        }

        // WebPage (sempre - con fallback sicuro)
        $webpage = $this->generate_webpage_schema($post_id);
        if ($webpage) {
            $schemas[] = $webpage;
        } else {
            // Fallback minimo
            $permalink = get_permalink($post_id);
            if ($permalink) {
                $schemas[] = [
                    '@type' => 'WebPage',
                    '@id' => esc_url($permalink),
                    'url' => esc_url($permalink),
                    'name' => get_the_title($post_id)
                ];
            }
        }

        // Article (solo per post)
        if (get_post_type($post_id) === 'post') {
            $article = $this->generate_article_schema($post_id);
            if ($article) {
                $schemas[] = $article;
            }
        }

        // === IDENTITÀ: SEMPRE presenti (garantiti) ===

        // Person (autore)
        $author = $this->generate_author_schema($post_id);
        if ($author) {
            $schemas[] = $author;
        }

        // Organization (publisher) - sempre
        $publisher = $this->generate_publisher_schema();
        if ($publisher) {
            $schemas[] = $publisher;
        }

        // === OPZIONALI: solo se disponibili ===

        // ImageObject (featured image)
        $image = $this->generate_image_schema($post_id);
        if ($image) {
            $schemas[] = $image;
        }

        // BreadcrumbList (categoria o gerarchia)
        $breadcrumb = $this->generate_breadcrumb_schema($post_id);
        if ($breadcrumb) {
            $schemas[] = $breadcrumb;
        }

        // === VALIDAZIONE WHITELIST ===
        $schemas = array_filter($schemas, function($schema) {
            if (!is_array($schema) || !isset($schema['@type'])) {
                return false;
            }
            $type = $schema['@type'];
            if (!in_array($type, LBS_ALLOWED_SCHEMA_TYPES)) {
                return false;
            }
            // Protezione: @id deve essere valido
            if (isset($schema['@id']) && !$schema['@id']) {
                return false;
            }
            return true;
        });

        // === ANTI-DOPPIONI ===
        $schemas = array_filter($schemas);
        $schemas = array_values($schemas);

        // === GARANZIA MINIMA: almeno 3 schemi core ===
        if (count($schemas) < 3) {
            $permalink = get_permalink($post_id);
            if (!$permalink) return false;

            // Fallback minimo garantito
            $schemas = [
                [
                    '@type' => 'WebSite',
                    '@id' => home_url() . '#website',
                    'url' => home_url(),
                    'name' => get_bloginfo('name')
                ],
                [
                    '@type' => 'WebPage',
                    '@id' => esc_url($permalink),
                    'url' => esc_url($permalink),
                    'name' => get_the_title($post_id)
                ],
                [
                    '@type' => 'Organization',
                    '@id' => home_url() . '#organization',
                    'name' => get_bloginfo('name'),
                    'url' => home_url()
                ]
            ];
        }

        $schema_output = [
            '@context' => 'https://schema.org',
            '@graph' => $schemas
        ];

        return '<script type="application/ld+json">' .
            wp_json_encode($schema_output, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT) .
            '</script>';
    }

    /**
    * Schema.org per archivi tassonomie (CollectionPage)
    */
    private function output_taxonomy_schema($term) {
        $url = get_term_link($term);
        if (is_wp_error($url)) return;

        $schema = [
            '@context' => 'https://schema.org',
            '@graph' => [
                // WebSite
                [
                    '@type' => 'WebSite',
                    '@id' => home_url() . '#website',
                    'url' => home_url(),
                    'name' => get_bloginfo('name')
                ],
                // CollectionPage (per archivi tassonomie)
                [
                    '@type' => 'CollectionPage',
                    '@id' => esc_url($url),
                    'url' => esc_url($url),
                    'name' => $term->name,
                    'description' => !empty($term->description) ? wp_strip_all_tags($term->description) : '',
                    'isPartOf' => [
                        '@id' => home_url() . '#website'
                    ],
                    'about' => [
                        '@id' => home_url() . '#organization'
                    ]
                ],
                // Organization
                [
                    '@type' => 'Organization',
                    '@id' => home_url() . '#organization',
                    'name' => get_bloginfo('name'),
                    'url' => home_url()
                ]
            ]
        ];

        echo '<script type="application/ld+json">' .
            wp_json_encode($schema, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT) .
            '</script>';
    }

    // ============================================================================
    // SCHEMA: GENERATORI CORE
    // ============================================================================

    /**
    * Genera schema WebSite
    */
    private function generate_website_schema() {
        $home_url = home_url();
        if (!$home_url) {
            return false;
        }

        return [
            '@type' => 'WebSite',
            '@id' => esc_url($home_url) . '#website',
            'url' => esc_url($home_url),
            'name' => get_bloginfo('name'),
            'description' => get_bloginfo('description'),
            'publisher' => [
                '@id' => esc_url($home_url) . '#organization'
            ],
            'about' => [
                '@id' => esc_url($home_url) . '#organization'
            ],
            'potentialAction' => [
                '@type' => 'SearchAction',
                'target' => esc_url($home_url) . '?s={search_term_string}',
                'query-input' => 'required name=search_term_string'
            ]
        ];
    }

    /**
    * Genera schema WebPage
    */
    private function generate_webpage_schema($post_id) {
        $post = get_post($post_id);
        if (!$post) {
            return false;
        }

        $permalink = get_permalink($post_id);
        if (!$permalink) {
            return false;
        }

        $schema = [
            '@type' => 'WebPage',
            '@id' => esc_url($permalink) . '#webpage',
            'url' => esc_url($permalink),
            'name' => get_the_title($post_id),
            'isPartOf' => [
                '@id' => home_url() . '#website'
            ],
            'datePublished' => get_the_date('c', $post_id),
            'dateModified' => get_the_modified_date('c', $post_id),
            'inLanguage' => get_locale(),
            'potentialAction' => [
                '@type' => 'ReadAction',
                'target' => esc_url($permalink)
            ]
        ];

        // Primary image (se esiste)
        $image_schema = $this->generate_image_schema($post_id);
        if ($image_schema && isset($image_schema['@id'])) {
            $schema['primaryImageOfPage'] = [
                '@id' => $image_schema['@id']
            ];
        }

        // Description
        $excerpt = get_the_excerpt($post_id);
        if (!empty($excerpt)) {
            $schema['description'] = wp_strip_all_tags($excerpt);
        }

        return $schema;
    }

    /**
    * Genera schema Article (solo per post)
    */
    private function generate_article_schema($post_id) {
        $post = get_post($post_id);
        if (!$post) {
            return false;
        }

        $permalink = get_permalink($post_id);
        if (!$permalink) {
            return false;
        }

        $schema = [
            '@type' => 'Article',
            '@id' => esc_url($permalink) . '#article',
            'isPartOf' => [
                '@id' => esc_url($permalink) . '#webpage'
            ],
            'headline' => get_the_title($post_id),
            'datePublished' => get_the_date('c', $post_id),
            'dateModified' => get_the_modified_date('c', $post_id),
            'author' => [
                '@id' => get_author_posts_url($post->post_author) . '#author'
            ],
            'publisher' => [
                '@id' => home_url() . '#organization'
            ],
            'mainEntityOfPage' => [
                '@id' => esc_url($permalink) . '#webpage'
            ],
            'inLanguage' => get_locale()
        ];

        // Featured image
        if (has_post_thumbnail($post_id)) {
            $image_schema = $this->generate_image_schema($post_id);
            if ($image_schema) {
                $schema['thumbnailUrl'] = $image_schema['url'] ?? '';
                if (isset($image_schema['@id'])) {
                    $schema['image'] = ['@id' => $image_schema['@id']];
                }
            }
        }

        // Description
        $excerpt = get_the_excerpt($post_id);
        if (!empty($excerpt)) {
            $schema['description'] = wp_strip_all_tags($excerpt);
        }

        // Reading time (future-proofing per assistant vocali)
        $word_count = str_word_count(strip_tags($post->post_content));
        $reading_time_minutes = max(1, round($word_count / 200));
        $schema['timeRequired'] = 'PT' . $reading_time_minutes . 'M';

        // Keywords da tag
        $tags = get_the_tags($post_id);
        if ($tags && !is_wp_error($tags)) {
            $keywords = wp_list_pluck($tags, 'name');
            $schema['keywords'] = implode(', ', $keywords);
        }

        // Speakable (per Google Assistant e dispositivi vocali)
        $schema['speakable'] = [
            '@type' => 'SpeakableSpecification',
            'xpath' => [
                '/html/head/title',
                '/html/body//article//h1'
            ]
        ];

        // ArticleSection (categoria principale o più profonda)
        $categories = get_the_category($post_id);
        if (!empty($categories)) {
            // Supporto Yoast Primary Term (se disponibile)
            $primary_term_fn = 'yoast_get_primary_term_id';
            if (function_exists($primary_term_fn)) {
                $primary = call_user_func($primary_term_fn, 'category', $post_id);
                if ($primary) {
                    $category = get_term($primary, 'category');
                } else {
                    $category = $this->get_deepest_category($categories);
                }
            } else {
                $category = $this->get_deepest_category($categories);
            }

            if ($category && !is_wp_error($category)) {
                $schema['articleSection'] = $category->name;
            }
        }

        return $schema;
    }

    // ============================================================================
    // SCHEMA: GENERATORI IDENTITÀ
    // ============================================================================

    /**
    * Genera schema Person (autore)
    */
    private function generate_author_schema($post_id) {
        $post = get_post($post_id);
        if (!$post) {
            return false;
        }

        $author_id = $post->post_author;
        $author_url = get_author_posts_url($author_id);
        if (!$author_url) {
            return false;
        }

        $schema = [
            '@type' => 'Person',
            '@id' => esc_url($author_url) . '#author',
            'name' => get_the_author_meta('display_name', $author_id),
            'url' => esc_url($author_url),
            'sameAs' => [esc_url($author_url)],
            'worksFor' => [
                '@id' => home_url() . '#organization'
            ]
        ];

        // Image come ImageObject
        $author_image = $this->get_author_image($author_id);
        if ($author_image) {
            $schema['image'] = [
                '@type' => 'ImageObject',
                'url' => esc_url($author_image)
            ];
        }

        // Description
        $author_desc = get_the_author_meta('description', $author_id);
        if (!empty($author_desc)) {
            $schema['description'] = wp_strip_all_tags($author_desc);
        }

        return $schema;
    }

    /**
    * Genera schema Organization (publisher)
    */
    private function generate_publisher_schema() {
        $home_url = home_url();
        if (!$home_url) {
            return false;
        }

        return [
            '@type' => 'Organization',
            '@id' => esc_url($home_url) . '#organization',
            'name' => get_bloginfo('name'),
            'url' => esc_url($home_url),
            'logo' => [
                '@type' => 'ImageObject',
                '@id' => esc_url($home_url) . '#logo',
                'url' => $this->get_site_logo_url(),
                'width' => 600,
                'height' => 60
            ],
            'sameAs' => $this->get_social_profiles()
        ];
    }

    // ============================================================================
    // SCHEMA: GENERATORI MEDIA & NAVIGAZIONE
    // ============================================================================

    /**
    * Genera schema ImageObject (featured image)
    */
    private function generate_image_schema($post_id) {
        if (!has_post_thumbnail($post_id)) {
            return false;
        }

        $thumbnail_id = get_post_thumbnail_id($post_id);
        $image_url = get_the_post_thumbnail_url($post_id, 'full');
        if (!$image_url) {
            return false;
        }

        $image_meta = wp_get_attachment_metadata($thumbnail_id);

        $schema = [
            '@type' => 'ImageObject',
            '@id' => esc_url($image_url) . '#image',
            'url' => esc_url($image_url)
        ];

        if (isset($image_meta['width'])) {
            $schema['width'] = absint($image_meta['width']);
        }

        if (isset($image_meta['height'])) {
            $schema['height'] = absint($image_meta['height']);
        }

        $alt_text = get_post_meta($thumbnail_id, '_wp_attachment_image_alt', true);
        if (!empty($alt_text)) {
            $schema['caption'] = sanitize_text_field($alt_text);
        }

        return $schema;
    }

    /**
    * Genera schema BreadcrumbList (categoria o gerarchia pagine)
    */
    private function generate_breadcrumb_schema($post_id) {
        $post = get_post($post_id);
        if (!$post) {
            return false;
        }

        $has_category = has_category('', $post_id);
        $has_parent = ($post->post_parent > 0);

        if (!$has_category && !$has_parent) {
            return false;
        }

        $items = [];
        $position = 1;

        // Home
        $items[] = [
            '@type' => 'ListItem',
            'position' => $position++,
            'item' => [
                '@type' => 'WebPage',
                '@id' => home_url(),
                'name' => __('Home', 'lean-bunker-seo')
            ]
        ];

        // Categoria o parent
        if ($has_category && get_post_type($post_id) === 'post') {
            $categories = get_the_category($post_id);
            if (!empty($categories)) {
                // Supporto Yoast Primary Term (se disponibile)
                $primary_term_fn = 'yoast_get_primary_term_id';
                if (function_exists($primary_term_fn)) {
                    $primary = call_user_func($primary_term_fn, 'category', $post_id);
                    if ($primary) {
                        $category = get_term($primary, 'category');
                    } else {
                        $category = $this->get_deepest_category($categories);
                    }
                } else {
                    $category = $this->get_deepest_category($categories);
                }

                if ($category && !is_wp_error($category)) {
                    $items[] = [
                        '@type' => 'ListItem',
                        'position' => $position++,
                        'item' => [
                            '@type' => 'WebPage',
                            '@id' => get_category_link($category->term_id),
                            'name' => $category->name
                        ]
                    ];
                }
            }
        } elseif ($has_parent) {
            $parent_hierarchy = $this->get_page_hierarchy($post_id);
            foreach ($parent_hierarchy as $parent_page) {
                $items[] = [
                    '@type' => 'ListItem',
                    'position' => $position++,
                    'item' => [
                        '@type' => 'WebPage',
                        '@id' => get_permalink($parent_page->ID),
                        'name' => $parent_page->post_title
                    ]
                ];
            }
        }

        // Pagina corrente
        $permalink = get_permalink($post_id);
        if ($permalink) {
            $items[] = [
                '@type' => 'ListItem',
                'position' => $position,
                'item' => [
                    '@type' => 'WebPage',
                    '@id' => esc_url($permalink),
                    'name' => get_the_title($post_id)
                ]
            ];
        }

        return [
            '@type' => 'BreadcrumbList',
            '@id' => esc_url($permalink) . '#breadcrumb',
            'itemListElement' => $items
        ];
    }

    // ============================================================================
    // SCHEMA: HELPER FUNCTIONS
    // ============================================================================

    /**
    * Ottiene la categoria più profonda (per gerarchie nidificate)
    */
    private function get_deepest_category($categories) {
        $deepest = null;
        $max_depth = 0;

        foreach ($categories as $cat) {
            if (is_wp_error($cat)) continue;
            $depth = $this->get_category_depth($cat);
            if ($depth > $max_depth) {
                $max_depth = $depth;
                $deepest = $cat;
            }
        }

        return $deepest ?: $categories[0];
    }

    /**
    * Calcola la profondità di una categoria nella gerarchia
    */
    private function get_category_depth($category) {
        if (is_wp_error($category)) return 0;
        $depth = 0;
        while ($category->parent) {
            $depth++;
            $category = get_term($category->parent, 'category');
            if (is_wp_error($category)) break;
        }
        return $depth;
    }

    /**
    * Ottiene la gerarchia di pagine parent (per pagine nidificate)
    */
    private function get_page_hierarchy($post_id) {
        $hierarchy = [];
        $post = get_post($post_id);
        while ($post && $post->post_parent) {
            $parent = get_post($post->post_parent);
            if (!$parent || is_wp_error($parent)) break;
            array_unshift($hierarchy, $parent);
            $post = $parent;
        }
        return $hierarchy;
    }

    /**
    * Ottiene l'URL del logo del sito (custom logo > site icon > fallback)
    */
    private function get_site_logo_url() {
        $custom_logo_id = get_theme_mod('custom_logo');
        if ($custom_logo_id) {
            $logo = wp_get_attachment_image_src($custom_logo_id, 'full');
            if ($logo && !is_wp_error($logo)) {
                return esc_url($logo[0]);
            }
        }

        $site_icon = get_site_icon_url(512);
        if ($site_icon) {
            return esc_url($site_icon);
        }

        return esc_url(home_url('/wp-content/uploads/logo.png'));
    }

    /**
    * Ottiene l'immagine dell'autore (escludendo Gravatar di default)
    */
    private function get_author_image($author_id) {
        $avatar_url = get_avatar_url($author_id, ['size' => 250]);
        if ($avatar_url && strpos($avatar_url, 'gravatar.com') === false) {
            return esc_url($avatar_url);
        }
        return false;
    }

    /**
    * Ottiene i profili social (da Yoast o theme mods)
    */
    private function get_social_profiles() {
        $profiles = [];

        // Supporto Yoast SEO (se disponibile)
        $yoast_fn = 'YoastSEO';
        if (function_exists($yoast_fn)) {
            $yoast = call_user_func($yoast_fn);
            if ($yoast && isset($yoast->meta)) {
                $social = $yoast->meta->for_current_page()->twitter_social_profiles;
                if (!empty($social)) {
                    return array_map('esc_url', $social);
                }
            }
        }

        // Fallback: theme mods
        $social_urls = [
            get_theme_mod('facebook_url'),
            get_theme_mod('twitter_url'),
            get_theme_mod('linkedin_url'),
            get_theme_mod('instagram_url')
        ];

        foreach ($social_urls as $url) {
            if (!empty($url)) {
                $profiles[] = esc_url($url);
            }
        }

        return $profiles;
    }

    // ============================================================================
    // BREADCRUMB SHORTCODE
    // ============================================================================

    /**
    * Shortcode breadcrumb [lb_breadcrumb]
    */
    public function shortcode_breadcrumb($atts = []) {
        $atts = shortcode_atts([
            'home_label' => __('Home', 'lean-bunker-seo'),
            'separator' => '&raquo;',
            'wrap_before' => '<nav class="lb-breadcrumb" aria-label="Breadcrumb">',
            'wrap_after' => '</nav>',
            'before' => '',
            'after' => '',
            'show_schema' => 'true'
        ], $atts, 'lb_breadcrumb');

        $items = $this->get_breadcrumb_items();
        if (empty($items)) {
            return '';
        }

        $output = $atts['wrap_before'];
        $output .= $atts['before'];

        $count = count($items);
        $position = 1;

        foreach ($items as $item) {
            $is_last = ($position === $count);

            if ($is_last) {
                // Ultimo elemento (corrente) - non linkato
                $output .= '<span class="lb-breadcrumb-current">' . esc_html($item['name']) . '</span>';
            } else {
                // Elementi precedenti - linkati
                $output .= '<a href="' . esc_url($item['url']) . '">' . esc_html($item['name']) . '</a>';
            }

            // Separatore (tranne dopo l'ultimo)
            if (!$is_last) {
                $output .= ' <span class="lb-breadcrumb-separator">' . $atts['separator'] . '</span> ';
            }

            $position++;
        }

        $output .= $atts['after'];
        $output .= $atts['wrap_after'];

        // Schema.org BreadcrumbList (se richiesto)
        if ($atts['show_schema'] === 'true') {
            $schema = $this->generate_breadcrumb_schema_from_items($items);
            if ($schema) {
                $output .= $schema;
            }
        }

        return $output;
    }

    /**
    * Ottiene gli items del breadcrumb
    */
    private function get_breadcrumb_items() {
        $items = [];

        // Home
        $items[] = [
            'name' => __('Home', 'lean-bunker-seo'),
            'url' => home_url()
        ];

        // Frontend logic
        if (is_front_page() || is_home()) {
            // Solo home
            return $items;
        }

        if (is_singular()) {
            global $post;

            // Categorie per post
            if (get_post_type() === 'post' && has_category()) {
                $categories = get_the_category($post->ID);
                if (!empty($categories)) {
                    // Usa categoria primaria o più profonda
                    $primary_term_fn = 'yoast_get_primary_term_id';
                    if (function_exists($primary_term_fn)) {
                        $primary = call_user_func($primary_term_fn, 'category', $post->ID);
                        if ($primary) {
                            $category = get_term($primary, 'category');
                        } else {
                            $category = $this->get_deepest_category($categories);
                        }
                    } else {
                        $category = $this->get_deepest_category($categories);
                    }

                    if ($category && !is_wp_error($category)) {
                        // Aggiungi gerarchia categoria
                        $category_hierarchy = $this->get_category_hierarchy($category);
                        foreach ($category_hierarchy as $cat) {
                            $items[] = [
                                'name' => $cat->name,
                                'url' => get_category_link($cat->term_id)
                            ];
                        }
                    }
                }
            }
            // Parent pages per pagine
            elseif ($post->post_parent) {
                $parent_hierarchy = $this->get_page_hierarchy($post->ID);
                foreach ($parent_hierarchy as $parent) {
                    $items[] = [
                        'name' => $parent->post_title,
                        'url' => get_permalink($parent->ID)
                    ];
                }
            }

            // Post corrente
            $items[] = [
                'name' => get_the_title($post->ID),
                'url' => get_permalink($post->ID)
            ];
        }
        elseif (is_category() || is_tag() || is_tax()) {
            $term = get_queried_object();

            if (is_category() && $term->parent) {
                // Gerarchia categoria
                $parent_hierarchy = $this->get_category_hierarchy($term);
                array_pop($parent_hierarchy); // Rimuovi l'ultimo (è la categoria corrente)
                foreach ($parent_hierarchy as $parent) {
                    $items[] = [
                        'name' => $parent->name,
                        'url' => get_category_link($parent->term_id)
                    ];
                }
            }

            // Termine corrente
            $items[] = [
                'name' => $term->name,
                'url' => get_term_link($term)
            ];
        }
        elseif (is_author()) {
            $author = get_queried_object();
            $items[] = [
                'name' => $author->display_name,
                'url' => get_author_posts_url($author->ID)
            ];
        }
        elseif (is_date()) {
            if (is_year()) {
                $items[] = [
                    'name' => get_the_date('Y'),
                    'url' => get_year_link(get_query_var('year'))
                ];
            }
            elseif (is_month()) {
                $items[] = [
                    'name' => get_the_date('F Y'),
                    'url' => get_month_link(get_query_var('year'), get_query_var('monthnum'))
                ];
            }
            elseif (is_day()) {
                $items[] = [
                    'name' => get_the_date('F j, Y'),
                    'url' => get_day_link(get_query_var('year'), get_query_var('monthnum'), get_query_var('day'))
                ];
            }
        }
        elseif (is_search()) {
            $items[] = [
                'name' => sprintf(__('Risultati per: %s', 'lean-bunker-seo'), get_search_query()),
                'url' => ''
            ];
        }
        elseif (is_404()) {
            $items[] = [
                'name' => __('Pagina non trovata', 'lean-bunker-seo'),
                'url' => ''
            ];
        }

        return $items;
    }

    /**
    * Ottiene la gerarchia completa di una categoria
    */
    private function get_category_hierarchy($category) {
        $hierarchy = [];

        // Risali la gerarchia
        $current = $category;
        while ($current && !is_wp_error($current)) {
            array_unshift($hierarchy, $current);
            if ($current->parent) {
                $current = get_term($current->parent, 'category');
            } else {
                break;
            }
        }

        return $hierarchy;
    }

    /**
    * Schema.org per breadcrumb shortcode
    */
    private function generate_breadcrumb_schema_from_items($items) {
        if (count($items) < 2) {
            return '';
        }

        $list_items = [];
        $position = 1;

        foreach ($items as $item) {
            $list_items[] = [
                '@type' => 'ListItem',
                'position' => $position++,
                'item' => [
                    '@type' => 'WebPage',
                    '@id' => !empty($item['url']) ? esc_url($item['url']) : home_url(),
                    'name' => $item['name']
                ]
            ];
        }

        $schema = [
            '@type' => 'BreadcrumbList',
            '@id' => home_url() . '#breadcrumb',
            'itemListElement' => $list_items
        ];

        return '<script type="application/ld+json">' .
            wp_json_encode(['@context' => 'https://schema.org', '@graph' => [$schema]], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT) .
            '</script>';
    }

    // ============================================================================
    // ADMIN: UI & BULK OPERATIONS
    // ============================================================================

    /**
    * Aggiunge la pagina impostazioni nel menu Admin
    */
    public function admin_menu() {
        add_options_page(
            'Lean Bunker SEO',
            'Lean Bunker SEO',
            'manage_options',
            'lean-bunker-seo',
            [$this, 'admin_page']
        );
    }

    /**
    * Render pagina impostazioni
    */
    public function admin_page() {
        if (!current_user_can('manage_options')) return;

        if (isset($_POST['lb_seo_save'])) {
            check_admin_referer('lb_seo_nonce');

            update_option($this->option_prefix . 'og_fallback_image', esc_url_raw($_POST['og_fallback_image'] ?? ''));
            update_option($this->option_prefix . 'title_template', sanitize_text_field($_POST['title_template'] ?? '{title} - {sitename}'));
            update_option($this->option_prefix . 'noindex_archives', isset($_POST['noindex_archives']) ? '1' : '0');
            update_option($this->option_prefix . 'noindex_search', isset($_POST['noindex_search']) ? '1' : '0');
            update_option($this->option_prefix . 'nofollow_external', isset($_POST['nofollow_external']) ? '1' : '0');
            update_option($this->option_prefix . 'sitemap_ping_interval', sanitize_text_field($_POST['sitemap_ping_interval'] ?? ''));
            update_option($this->option_prefix . 'baidu_token', sanitize_text_field($_POST['baidu_token'] ?? ''));

            $this->reschedule_pinger();

            echo '<div class="notice notice-success"><p>✅ Impostazioni salvate.</p></div>';
        }

        $og_fallback = get_option($this->option_prefix . 'og_fallback_image');
        $title_template = get_option($this->option_prefix . 'title_template', '{title} - {sitename}');
        $noindex_archives = get_option($this->option_prefix . 'noindex_archives', '0');
        $noindex_search = get_option($this->option_prefix . 'noindex_search', '0');
        $nofollow_external = get_option($this->option_prefix . 'nofollow_external', '0');
        $ping_interval = get_option($this->option_prefix . 'sitemap_ping_interval', '');
        $baidu_token = get_option($this->option_prefix . 'baidu_token', '');

        // Ottieni log ping
        $ping_log = get_option('lbs_last_ping_log', []);
        $ping_stats = get_option('lbs_ping_stats', []);
        $next_scheduled = wp_next_scheduled('lean_bunker_sitemap_ping');
        ?>
        <div class="wrap" style="max-width:900px;">
            <h1>Lean Bunker SEO</h1>
            <div style="background:#f0f6ff; padding:15px; border-left:4px solid #2271b1; margin-bottom:20px;">
                <strong>✨ Plugin completamente autonomo</strong><br>
                Meta tags, Open Graph, Schema.org e breadcrumb generati nativamente da WordPress. Nessuna API esterna richiesta. Performance ottimizzata e privacy garantita.
            </div>

            <form method="post">
                <?php wp_nonce_field('lb_seo_nonce'); ?>

                <table class="form-table">
                    <tr>
                        <th scope="row">Open Graph Fallback Image URL</th>
                        <td>
                            <input type="url" name="og_fallback_image" value="<?php echo esc_attr($og_fallback); ?>" class="regular-text" placeholder="https://esempio.com/fallback.jpg">
                            <p class="description">Immagine fallback per Open Graph (1200×630 px).</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Title Template</th>
                        <td>
                            <input type="text" name="title_template" value="<?php echo esc_attr($title_template); ?>" class="regular-text" placeholder="{title} - {sitename}">
                            <p class="description">
                                Token disponibili: <code>{title}</code>, <code>{sitename}</code>, <code>{category}</code>, <code>{sep}</code><br>
                                Esempi: <code>{title} {sep} {sitename}</code> | <code>{sitename} {sep} {title}</code>
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Metodo generazione</th>
                        <td>
                            <strong style="color:#2271b1;">✅ WordPress Nativo</strong>
                            <p class="description">Meta title: Template configurabile<br>Meta description: Excerpt o primi 25-30 parole del contenuto<br>Schema.org: JSON-LD automatico per tutti i contenuti<br>Breadcrumb: Shortcode <code>[lb_breadcrumb]</code></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Noindex/Nofollow</th>
                        <td>
                            <label>
                                <input type="checkbox" name="noindex_archives" value="1" <?php checked($noindex_archives, '1'); ?>>
                                Noindex archivi (categorie, tag, autori, date)
                            </label><br>
                            <label>
                                <input type="checkbox" name="noindex_search" value="1" <?php checked($noindex_search, '1'); ?>>
                                Noindex pagina di ricerca
                            </label><br>
                            <label>
                                <input type="checkbox" name="nofollow_external" value="1" <?php checked($nofollow_external, '1'); ?>>
                                Nofollow link esterni (esclusi mailto: e tel:)
                            </label>
                            <p class="description">Controlli SEO avanzati. Utile per evitare contenuti duplicati.</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Shortcodes</th>
                        <td>
                            <code>[lb_seo_description]</code> - Mostra meta description<br>
                            <code>[lb_breadcrumb]</code> - Mostra breadcrumb navigazionale
                        </td>
                    </tr>
                </table>

                <hr style="margin:30px 0;">

                <h2>📡 Sitemap Auto-Ping (Bing, Yandex, Baidu + Google cortesia)</h2>

                <?php if (!empty($ping_log)): ?>
                    <div style="background:#f8f9f9; padding:15px; margin-bottom:20px; border-left:4px solid #11a0d2;">
                        <h3 style="margin-top:0;">📊 Stato Ping Attuale</h3>
                        <p><strong>Sitemap:</strong> <code><?php echo esc_html($ping_log['sitemap_url'] ?? home_url('/wp-sitemap.xml')); ?></code></p>
                        <p><strong>Ultimo ping:</strong> <?php echo date('d/m/Y H:i:s', $ping_log['timestamp'] ?? time()); ?></p>

                        <?php if (!empty($ping_log['results'])): ?>
                            <p><strong>Risultati:</strong></p>
                            <ul style="margin:10px 0; padding-left:20px;">
                                <?php foreach ($ping_log['results'] as $engine => $result): ?>
                                    <li>
                                        <?php
                                        $icon = ($result === 'success') ? '✅' : (($result === 'error') ? '❌' : '⚠️');
                                        $color = ($result === 'success') ? '#46b450' : (($result === 'error') ? '#dc3232' : '#dba617');
                                        ?>
                                        <span style="color:<?php echo $color; ?>;"><?php echo $icon; ?></span>
                                        <strong><?php echo ucfirst($engine); ?>:</strong>
                                        <?php
                                        if (is_array($result)) {
                                            echo $result['message'] ?? $result['status'] ?? 'Inviato';
                                        } else {
                                            echo ($result === 'success') ? 'Successo' : (($result === 'error') ? 'Errore' : 'Inviato');
                                        }
                                        ?>
                                        <?php if (!empty($ping_stats[$engine])): ?>
                                            <small style="color:#666;">(<?php echo $ping_stats[$engine]; ?> ping totali)</small>
                                        <?php endif; ?>
                                    </li>
                                <?php endforeach; ?>
                            </ul>
                        <?php endif; ?>

                        <?php if ($next_scheduled): ?>
                            <p><strong>Prossimo ping:</strong> <?php echo date('d/m/Y H:i:s', $next_scheduled); ?></p>
                        <?php else: ?>
                            <p style="color:#dc3232;"><strong>⚠️ Nessun ping schedulato. Configura la frequenza qui sotto.</strong></p>
                        <?php endif; ?>

                        <button type="button" id="lb-manual-ping" class="button button-secondary" style="margin-top:10px;">
                            <span class="dashicons dashicons-update"></span> Ping Sitemap Ora
                        </button>
                        <span id="lb-ping-status" style="margin-left:10px;"></span>
                    </div>
                <?php endif; ?>

                <table class="form-table">
                    <tr>
                        <th scope="row">Frequenza ping</th>
                        <td>
                            <select name="sitemap_ping_interval">
                                <option value="">Disabilitato</option>
                                <option value="hourly" <?php selected($ping_interval, 'hourly'); ?>>Ogni ora</option>
                                <option value="twicedaily" <?php selected($ping_interval, 'twicedaily'); ?>>Due volte al giorno</option>
                                <option value="daily" <?php selected($ping_interval, 'daily'); ?>>Ogni giorno</option>
                                <option value="weekly" <?php selected($ping_interval, 'weekly'); ?>>Ogni settimana</option>
                                <option value="monthly" <?php selected($ping_interval, 'monthly'); ?>>Ogni mese</option>
                            </select>
                            <p class="description">
                                Invia la sitemap nativa di WordPress (<code>/wp-sitemap.xml</code>) ai motori di ricerca.<br>
                                <strong style="color:#2271b1;">Nota:</strong> Google accetta il ping ma non garantisce l'indicizzazione. Usa Search Console per Google.
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Baidu Token (opzionale)</th>
                        <td>
                            <input type="text" name="baidu_token" value="<?php echo esc_attr($baidu_token); ?>" class="regular-text">
                            <p class="description">Token da <a href="https://ziyuan.baidu.com/" target="_blank">Baidu Webmaster</a>. Se vuoto, Baidu è disabilitato.</p>
                        </td>
                    </tr>
                </table>

                <p class="submit">
                    <input type="submit" name="lb_seo_save" class="button button-primary" value="Salva impostazioni">
                </p>

                <hr style="margin:30px 0;">

                <h2>🔄 Completa automaticamente i contenuti esistenti</h2>
                <p>Genera meta title, description e Open Graph per tutti i post, pagine, CPT e tassonomie che ne sono sprovvisti. <strong>Istantaneo, senza API esterne.</strong></p>
                <button id="lb-bulk-start" class="button button-secondary">Avvia completamento</button>
                <div id="lb-bulk-status" style="margin-top:10px; padding:10px; background:#f8f9f9; display:none;"></div>

                <script>
                jQuery(document).ready(function($) {
                    // Ping manuale
                    $('#lb-manual-ping').on('click', function() {
                        const $btn = $(this);
                        const $status = $('#lb-ping-status');
                        $btn.prop('disabled', true).html('<span class="dashicons dashicons-update" style="animation: spin 1s linear infinite;"></span> Invio in corso...');
                        $status.html('');

                        $.post(ajaxurl, {
                            action: 'lb_seo_manual_ping',
                            nonce: '<?php echo wp_create_nonce("lb_seo_manual_ping"); ?>'
                        }, function(res) {
                            $btn.prop('disabled', false).html('<span class="dashicons dashicons-update"></span> Ping Sitemap Ora');
                            if (res.success) {
                                $status.html('<span style="color:#46b450;">✅ Ping inviati con successo! Ricarica la pagina per vedere i risultati.</span>');
                                setTimeout(function() {
                                    location.reload();
                                }, 1500);
                            } else {
                                $status.html('<span style="color:#dc3232;">❌ Errore: ' + (res.data?.message || 'Impossibile inviare i ping') + '</span>');
                            }
                        }).fail(function() {
                            $btn.prop('disabled', false).html('<span class="dashicons dashicons-update"></span> Ping Sitemap Ora');
                            $status.html('<span style="color:#dc3232;">❌ Errore di connessione</span>');
                        });
                    });

                    // Bulk generation
                    $('#lb-bulk-start').on('click', function() {
                        const $btn = $(this);
                        const $status = $('#lb-bulk-status');
                        let processed = 0;
                        let processingTaxonomies = false;

                        $btn.prop('disabled', true).text('In corso...');
                        $status.show().html('Ricerca contenuti mancanti…');

                        function processNextBatch() {
                            $.post(ajaxurl, {
                                action: 'lb_seo_get_missing',
                                nonce: '<?php echo wp_create_nonce("lb_seo_bulk"); ?>'
                            }, function(res) {
                                if (!res.success || res.data.length === 0) {
                                    // Nessun post mancante, passa alle tassonomie
                                    processTaxonomies();
                                    return;
                                }

                                let completed = 0;
                                const totalInBatch = res.data.length;

                                res.data.forEach(id => {
                                    $.post(ajaxurl, {
                                        action: 'lb_seo_generate_single',
                                        post_id: id,
                                        nonce: '<?php echo wp_create_nonce("lb_seo_bulk"); ?>'
                                    }).always(() => {
                                        completed++;
                                        processed++;
                                        $status.html(`Processati: ${processed} &nbsp; | &nbsp; Batch corrente: ${completed}/${totalInBatch}`);
                                        if (completed === totalInBatch) {
                                            // Timeout più lungo per siti grandi (220.000+ articoli)
                                            setTimeout(processNextBatch, 300);
                                        }
                                    });
                                });
                            }).fail(() => {
                                $status.html('❌ Errore durante il recupero dei dati.');
                                $btn.prop('disabled', false).text('Riprova');
                            });
                        }

                        function processTaxonomies() {
                            processingTaxonomies = true;
                            $.post(ajaxurl, {
                                action: 'lb_seo_get_missing_taxonomies',
                                nonce: '<?php echo wp_create_nonce("lb_seo_bulk"); ?>'
                            }, function(res) {
                                if (!res.success || res.data.length === 0) {
                                    $status.html('✅ Completamento terminato! ' + processed + ' elementi aggiornati.');
                                    $btn.text('Fatto!');
                                    return;
                                }

                                let completed = 0;
                                const totalInBatch = res.data.length;

                                res.data.forEach(term => {
                                    $.post(ajaxurl, {
                                        action: 'lb_seo_generate_taxonomy_single',
                                        term_id: term.id,
                                        taxonomy: term.taxonomy,
                                        nonce: '<?php echo wp_create_nonce("lb_seo_bulk"); ?>'
                                    }).always(() => {
                                        completed++;
                                        processed++;
                                        $status.html(`Processati: ${processed} &nbsp; | &nbsp; Batch corrente: ${completed}/${totalInBatch}`);
                                        if (completed === totalInBatch) {
                                            setTimeout(processTaxonomies, 300);
                                        }
                                    });
                                });
                            }).fail(() => {
                                $status.html('❌ Errore durante il recupero delle tassonomie.');
                                $btn.prop('disabled', false).text('Riprova');
                            });
                        }

                        processNextBatch();
                    });
                });
                </script>
            </form>
        </div>
        <style>
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        </style>
        <?php
    }

    /**
    * Enqueue jQuery per admin
    */
    public function enqueue_scripts($hook) {
        if ('post.php' !== $hook && 'post-new.php' !== $hook && 'settings_page_lean-bunker-seo' !== $hook) return;
        wp_enqueue_script('jquery');
    }

    /**
    * Aggiunge metabox SEO su tutti i post type pubblici
    */
    public function add_metabox() {
        $types = get_post_types(['public' => true], 'names');
        foreach ($types as $type) {
            if ($type === 'attachment') continue;
            add_meta_box(
                'lb_seo_metabox',
                'Lean Bunker SEO',
                [$this, 'metabox_html'],
                $type,
                'normal',
                'high'
            );
        }
    }

    /**
    * Render metabox SEO nell'editor
    */
    public function metabox_html($post) {
        $title = get_post_meta($post->ID, '_lb_seo_title', true);
        $desc  = get_post_meta($post->ID, '_lb_seo_desc', true);
        $robots = get_post_meta($post->ID, '_lb_seo_robots', true);
        ?>
        <div style="background:#f8f9f9; padding:12px; border-left:4px solid #2271b1;">
            <p><strong>Meta Title (max 60)</strong><br>
            <input type="text" name="_lb_seo_title" value="<?php echo esc_attr($title); ?>" style="width:100%; max-width:600px;" maxlength="60"></p>

            <p><strong>Meta Description (max 155)</strong><br>
            <textarea name="_lb_seo_desc" rows="3" style="width:100%; max-width:600px;" maxlength="155"><?php echo esc_textarea($desc); ?></textarea></p>

            <p><strong>Meta Robots</strong><br>
            <select name="_lb_seo_robots" style="width:100%; max-width:600px;">
                <option value="">Default (index, follow)</option>
                <option value="noindex,follow" <?php selected($robots, 'noindex,follow'); ?>>Noindex, follow</option>
                <option value="noindex,nofollow" <?php selected($robots, 'noindex,nofollow'); ?>>Noindex, nofollow</option>
                <option value="index,nofollow" <?php selected($robots, 'index,nofollow'); ?>>Index, nofollow</option>
            </select>
            <br><small>Controlla indicizzazione e follow dei link per questo contenuto specifico.</small></p>

            <?php wp_nonce_field('lb_seo_save_meta', '_lb_seo_nonce'); ?>
            <button type="button" id="lb-generate-seo" class="button button-secondary" style="margin-top:8px;">Genera meta tag</button>
            <span id="lb-seo-status" style="margin-left:10px; color:#2271b1;"></span>
        </div>

        <script>
        jQuery(document).ready(function($) {
            $('#lb-generate-seo').on('click', function() {
                $('#lb-seo-status').text('Generazione in corso…');
                $.post(ajaxurl, {
                    action: 'lb_seo_generate',
                    post_id: <?php echo (int)$post->ID; ?>,
                    nonce: '<?php echo wp_create_nonce("lb_seo_generate"); ?>'
                }, function(res) {
                    if (res.success) {
                        $('input[name="_lb_seo_title"]').val(res.data.title);
                        $('textarea[name="_lb_seo_desc"]').val(res.data.desc);
                        $('#lb-seo-status').text('✅ Generato!');
                    } else {
                        $('#lb-seo-status').text('❌ Errore');
                    }
                });
            });
        });
        </script>
        <?php
    }

    /**
    * Metabox SEO per tassonomie (categorie, tag)
    */
    public function taxonomy_metabox_html($term, $taxonomy) {
        $title = get_term_meta($term->term_id, '_lb_seo_title', true);
        $desc = get_term_meta($term->term_id, '_lb_seo_desc', true);
        $robots = get_term_meta($term->term_id, '_lb_seo_robots', true);
        ?>
        <tr class="form-field">
            <th scope="row"><label for="lb_seo_title">Lean Bunker SEO</label></th>
            <td>
                <div style="background:#f8f9f9; padding:15px; border-left:4px solid #2271b1; margin-top:10px;">
                    <p><strong>Meta Title (max 60)</strong><br>
                    <input type="text" id="lb_seo_title" name="_lb_seo_title" value="<?php echo esc_attr($title); ?>" style="width:100%;" maxlength="60"></p>

                    <p><strong>Meta Description (max 155)</strong><br>
                    <textarea id="lb_seo_desc" name="_lb_seo_desc" rows="3" style="width:100%;" maxlength="155"><?php echo esc_textarea($desc); ?></textarea></p>

                    <p><strong>Meta Robots</strong><br>
                    <select name="_lb_seo_robots" style="width:100%;">
                        <option value="">Default (index, follow)</option>
                        <option value="noindex,follow" <?php selected($robots, 'noindex,follow'); ?>>Noindex, follow</option>
                        <option value="noindex,nofollow" <?php selected($robots, 'noindex,nofollow'); ?>>Noindex, nofollow</option>
                    </select>
                    <br><small>Controlla indicizzazione di questa tassonomia.</small></p>

                    <?php wp_nonce_field('lb_seo_save_taxonomy', '_lb_seo_nonce_taxonomy'); ?>
                </div>
            </td>
        </tr>
        <?php
    }

    /**
    * Salva meta SEO su save_post con verifica nonce
    */
    public function save_meta($post_id, $post) {
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
        if (!current_user_can('edit_post', $post_id)) return;
        if ($post->post_type === 'revision') return;

        // Verifica nonce
        if (!isset($_POST['_lb_seo_nonce']) || !wp_verify_nonce($_POST['_lb_seo_nonce'], 'lb_seo_save_meta')) {
            return;
        }

        if (isset($_POST['_lb_seo_title'])) {
            update_post_meta($post_id, '_lb_seo_title', sanitize_text_field($_POST['_lb_seo_title']));
        }

        if (isset($_POST['_lb_seo_desc'])) {
            update_post_meta($post_id, '_lb_seo_desc', sanitize_textarea_field($_POST['_lb_seo_desc']));
        }

        if (isset($_POST['_lb_seo_robots'])) {
            $allowed_values = ['', 'noindex,follow', 'noindex,nofollow', 'index,nofollow'];
            $robots = in_array($_POST['_lb_seo_robots'], $allowed_values) ? $_POST['_lb_seo_robots'] : '';
            update_post_meta($post_id, '_lb_seo_robots', sanitize_text_field($robots));
        }
    }

    /**
    * Salva meta SEO per tassonomie con verifica nonce
    */
    public function save_taxonomy_meta($term_id) {
        // Verifica nonce
        if (!isset($_POST['_lb_seo_nonce_taxonomy']) || !wp_verify_nonce($_POST['_lb_seo_nonce_taxonomy'], 'lb_seo_save_taxonomy')) {
            return;
        }

        if (isset($_POST['_lb_seo_title'])) {
            update_term_meta($term_id, '_lb_seo_title', sanitize_text_field($_POST['_lb_seo_title']));
        }

        if (isset($_POST['_lb_seo_desc'])) {
            update_term_meta($term_id, '_lb_seo_desc', sanitize_textarea_field($_POST['_lb_seo_desc']));
        }

        if (isset($_POST['_lb_seo_robots'])) {
            $allowed = ['', 'noindex,follow', 'noindex,nofollow'];
            $robots = in_array($_POST['_lb_seo_robots'], $allowed) ? $_POST['_lb_seo_robots'] : '';
            update_term_meta($term_id, '_lb_seo_robots', sanitize_text_field($robots));
        }
    }

    /**
    * Genera automaticamente meta tags alla prima pubblicazione
    */
    public function auto_generate_on_publish($new_status, $old_status, $post) {
        if ($new_status === 'publish' && $old_status !== 'publish' && !wp_is_post_revision($post)) {
            $has_title = get_post_meta($post->ID, '_lb_seo_title', true);
            $has_desc  = get_post_meta($post->ID, '_lb_seo_desc', true);

            if (!$has_title || !$has_desc) {
                $this->generate_for_post($post->ID);
            }
        }
    }

    // ============================================================================
    // GENERAZIONE NATIVA (ZERO API) - CON PROTEZIONE DATI MANUALI
    // ============================================================================

    /**
    * Genera meta title e description nativamente (senza API)
    * NON sovrascrive mai ciò che esiste già
    */
    private function generate_for_post($post_id) {
        $post = get_post($post_id);
        if (!$post) return ['title' => '', 'desc' => ''];

        // Verifica se esistono già
        $has_title = get_post_meta($post_id, '_lb_seo_title', true);
        $has_desc = get_post_meta($post_id, '_lb_seo_desc', true);

        // NON sovrascrivere mai ciò che esiste già
        if ($has_title && $has_desc) {
            return ['title' => $has_title, 'desc' => $has_desc];
        }

        // Meta Title: Usa template globale (solo se manca)
        if (!$has_title) {
            $template = get_option($this->option_prefix . 'title_template', '{title} - {sitename}');
            $post_title = wp_strip_all_tags($post->post_title);
            $site_name = get_bloginfo('name');
            $sep = apply_filters('document_title_separator', '-');
            $category = '';

            if (get_post_type($post_id) === 'post') {
                $categories = get_the_category($post_id);
                if (!empty($categories)) {
                    $category = $categories[0]->name;
                }
            }

            $title = str_replace('{title}', $post_title, $template);
            $title = str_replace('{sitename}', $site_name, $title);
            $title = str_replace('{category}', $category, $title);
            $title = str_replace('{sep}', $sep, $title);
            $title = trim($title);

            // FAIL-SAFE: se ci sono ancora token non sostituiti, usa fallback
            if (strpos($title, '{') !== false) {
                $title = $post_title . ' - ' . $site_name;
            }

            if (mb_strlen($title) > 60) {
                $title = mb_substr($title, 0, 57) . '...';
            }

            $title = sanitize_text_field($title);
        } else {
            $title = $has_title; // Mantieni quello esistente
        }

        // Meta Description: Excerpt o primi 28-30 parole (solo se manca)
        if (!$has_desc) {
            $desc = $post->post_excerpt;
            if (!$desc && !empty($post->post_content)) {
                $desc = wp_trim_words(wp_strip_all_tags($post->post_content), 28, '...');
            }
            if (!$desc) {
                $desc = wp_trim_words(wp_strip_all_tags($post->post_title), 15, '...');
            }
            if (mb_strlen($desc) > 155) {
                $desc = mb_substr($desc, 0, 152) . '...';
            }
            $desc = sanitize_textarea_field($desc);
        } else {
            $desc = $has_desc; // Mantieni quella esistente
        }

        // Salva solo se generato nuovo
        if (!$has_title) update_post_meta($post_id, '_lb_seo_title', $title);
        if (!$has_desc) update_post_meta($post_id, '_lb_seo_desc', $desc);

        return ['title' => $title, 'desc' => $desc];
    }

    /**
    * Genera meta title e description per tassonomie
    * NON sovrascrive mai ciò che esiste già
    */
    private function generate_for_taxonomy($term_id, $taxonomy) {
        $term = get_term($term_id, $taxonomy);
        if (!$term || is_wp_error($term)) return;

        // Verifica se esistono già
        $has_title = get_term_meta($term_id, '_lb_seo_title', true);
        $has_desc = get_term_meta($term_id, '_lb_seo_desc', true);

        // NON sovrascrivere mai ciò che esiste già
        if ($has_title && $has_desc) {
            return;
        }

        // Meta Title: Nome tassonomia + nome sito (solo se manca)
        if (!$has_title) {
            $site_name = get_bloginfo('name');
            $title = $term->name . ' - ' . $site_name;
            if (mb_strlen($title) > 60) {
                $title = mb_substr($title, 0, 57) . '...';
            }
            $title = sanitize_text_field($title);
        } else {
            $title = $has_title;
        }

        // Meta Description: Descrizione tassonomia o fallback (solo se manca)
        if (!$has_desc) {
            $desc = !empty($term->description) ? wp_strip_all_tags($term->description) : '';
            if (empty($desc)) {
                // Fallback: "Tutti gli articoli nella categoria X"
                $desc = sprintf(
                    __('Tutti gli articoli nella categoria %s', 'lean-bunker-seo'),
                    $term->name
                );
            }
            if (mb_strlen($desc) > 155) {
                $desc = mb_substr($desc, 0, 152) . '...';
            }
            $desc = sanitize_textarea_field($desc);
        } else {
            $desc = $has_desc;
        }

        // Salva solo se generato nuovo
        if (!$has_title) update_term_meta($term_id, '_lb_seo_title', $title);
        if (!$has_desc) update_term_meta($term_id, '_lb_seo_desc', $desc);
    }

    // ============================================================================
    // AJAX HANDLERS - CON LOCK PER EVITARE RACE CONDITION
    // ============================================================================

    /**
    * AJAX: Genera meta tags per un singolo post
    */
    public function ajax_generate() {
        check_ajax_referer('lb_seo_generate', 'nonce');
        if (!current_user_can('edit_post', $_POST['post_id'] ?? 0)) wp_die();

        $result = $this->generate_for_post(intval($_POST['post_id']));
        wp_send_json_success($result);
    }

    /**
    * AJAX: Ottiene post con meta tags mancanti (batch da 10)
    * Con lock per evitare race condition su siti grandi
    */
    public function ajax_get_missing_posts() {
        check_ajax_referer('lb_seo_bulk', 'nonce');
        if (!current_user_can('manage_options')) wp_die();

        // Lock per evitare esecuzioni parallele
        if (get_transient('lb_seo_bulk_lock')) {
            wp_send_json_error('Bulk generation già in corso. Attendi...');
        }

        set_transient('lb_seo_bulk_lock', 1, 120); // Lock per 2 minuti

        $post_types = get_post_types(['public' => true], 'names');
        unset($post_types['attachment']);

        $query = new WP_Query([
            'post_type'      => array_values($post_types),
            'post_status'    => 'publish',
            'posts_per_page' => 10,
            'fields'         => 'ids',
            'meta_query'     => [
                'relation' => 'OR',
                [
                    'key'     => '_lb_seo_title',
                    'value'   => '',
                    'compare' => 'NOT EXISTS'
                ],
                [
                    'key'     => '_lb_seo_title',
                    'value'   => '',
                    'compare' => '='
                ],
                [
                    'key'     => '_lb_seo_desc',
                    'value'   => '',
                    'compare' => 'NOT EXISTS'
                ],
                [
                    'key'     => '_lb_seo_desc',
                    'value'   => '',
                    'compare' => '='
                ]
            ]
        ]);

        wp_send_json_success(array_map('intval', $query->posts));
    }

    /**
    * AJAX: Genera meta tags per un post specifico (bulk operation)
    */
    public function ajax_generate_single() {
        check_ajax_referer('lb_seo_bulk', 'nonce');
        if (!current_user_can('manage_options')) wp_die();

        $post_id = intval($_POST['post_id'] ?? 0);
        if (!$post_id) {
            delete_transient('lb_seo_bulk_lock'); // Rilascia lock in caso di errore
            wp_send_json_error('ID mancante');
        }

        $this->generate_for_post($post_id);

        // NOTA: Il lock viene mantenuto fino al completamento del batch
        // Per un sistema più sofisticato, potresti usare un heartbeat che
        // controlla se ci sono ancora batch in corso prima di rilasciare
        wp_send_json_success();
    }

    /**
    * AJAX: Ottiene tassonomie con meta tags mancanti (batch da 10)
    */
    public function ajax_get_missing_taxonomies() {
        check_ajax_referer('lb_seo_bulk', 'nonce');
        if (!current_user_can('manage_options')) wp_die();

        // Ottieni tutte le tassonomie pubbliche
        $taxonomies = get_taxonomies(['public' => true], 'names');
        $missing_terms = [];

        foreach ($taxonomies as $taxonomy) {
            // Query per termini senza meta SEO
            $terms = get_terms([
                'taxonomy' => $taxonomy,
                'hide_empty' => false,
                'number' => 10 - count($missing_terms), // Limita a 10 totali
                'meta_query' => [
                    'relation' => 'OR',
                    [
                        'key' => '_lb_seo_title',
                        'value' => '',
                        'compare' => 'NOT EXISTS'
                    ],
                    [
                        'key' => '_lb_seo_title',
                        'value' => '',
                        'compare' => '='
                    ],
                    [
                        'key' => '_lb_seo_desc',
                        'value' => '',
                        'compare' => 'NOT EXISTS'
                    ],
                    [
                        'key' => '_lb_seo_desc',
                        'value' => '',
                        'compare' => '='
                    ]
                ]
            ]);

            if (!empty($terms) && !is_wp_error($terms)) {
                foreach ($terms as $term) {
                    $missing_terms[] = [
                        'id' => $term->term_id,
                        'taxonomy' => $taxonomy,
                        'name' => $term->name
                    ];
                    if (count($missing_terms) >= 10) break 2;
                }
            }
        }

        wp_send_json_success($missing_terms);
    }

    /**
    * AJAX: Genera meta tags per una tassonomia specifica (bulk operation)
    */
    public function ajax_generate_taxonomy_single() {
        check_ajax_referer('lb_seo_bulk', 'nonce');
        if (!current_user_can('manage_options')) wp_die();

        $term_id = intval($_POST['term_id'] ?? 0);
        $taxonomy = sanitize_text_field($_POST['taxonomy'] ?? '');

        if (!$term_id || !$taxonomy) {
            wp_send_json_error('Dati mancanti');
        }

        $this->generate_for_taxonomy($term_id, $taxonomy);
        wp_send_json_success();
    }

    /**
    * AJAX: Trigger manuale del ping sitemap
    */
    public function ajax_manual_ping() {
        check_ajax_referer('lb_seo_manual_ping', 'nonce');
        if (!current_user_can('manage_options')) wp_die();

        // Verifica che la sitemap esista
        $sitemap_url = home_url('/wp-sitemap.xml');
        $response = wp_remote_head($sitemap_url, ['timeout' => 5, 'sslverify' => true]);

        if (is_wp_error($response) || wp_remote_retrieve_response_code($response) !== 200) {
            wp_send_json_error([
                'message' => 'Sitemap non disponibile: ' . $sitemap_url
            ]);
        }

        // Invia ping
        $this->ping_search_engines();

        wp_send_json_success([
            'message' => 'Ping inviati con successo!'
        ]);
    }

    /**
    * Shortcode: [lb_seo_description]
    * Mostra la meta description in pagina
    */
    public function shortcode_seo_desc() {
        if (!is_singular()) return '';

        $post_id = get_queried_object_id();
        if (!$post_id) return '';

        $desc = get_post_meta($post_id, '_lb_seo_desc', true);
        if (!$desc) {
            $post = get_post($post_id);
            $desc = $post->post_excerpt ?: wp_trim_words($post->post_content, 25, '...');
        }

        return '<div class="lb-seo-preview" style="font-size:0.9em; color:#555; padding:8px; background:#f9f9f9; border-left:3px solid #ccc;">' .
            esc_html($desc) .
            '</div>';
    }

    // ============================================================================
    // CRON: SITEMAP PING - CON LOGGING MIGLIORATO
    // ============================================================================

    /**
    * Registra l'hook per il sitemap ping
    */
    public function schedule_pinger() {
        add_action('lean_bunker_sitemap_ping', [$this, 'ping_search_engines']);
    }

    /**
    * Attivazione: schedula il ping
    */
    public function activate_pinger() {
        $this->reschedule_pinger();
    }

    /**
    * Disattivazione: rimuove lo schedule
    */
    public function deactivate_pinger() {
        wp_clear_scheduled_hook('lean_bunker_sitemap_ping');
    }

    /**
    * Reschedula il ping in base alle impostazioni
    */
    public function reschedule_pinger() {
        wp_clear_scheduled_hook('lean_bunker_sitemap_ping');
        $interval = get_option($this->option_prefix . 'sitemap_ping_interval');

        if ($interval && in_array($interval, ['hourly', 'twicedaily', 'daily', 'weekly', 'monthly'])) {
            wp_schedule_event(time(), $interval, 'lean_bunker_sitemap_ping');
        }
    }

    /**
    * Verifica se la sitemap è accessibile prima di pingare
    */
    private function is_sitemap_accessible() {
        $sitemap_url = home_url('/wp-sitemap.xml');
        $response = wp_remote_head($sitemap_url, [
            'timeout' => 3,
            'sslverify' => true
        ]);

        if (is_wp_error($response)) {
            return false;
        }

        $code = wp_remote_retrieve_response_code($response);
        return $code === 200;
    }

    /**
    * Invia la sitemap a Bing, Yandex, Baidu + Google (cortesia) con logging dettagliato
    */
    public function ping_search_engines() {
        // Verifica sitemap
        if (!$this->is_sitemap_accessible()) {
            $this->log_ping_error('Sitemap non accessibile');
            return;
        }

        $sitemap_url = home_url('/wp-sitemap.xml');
        $encoded_sitemap = urlencode($sitemap_url);

        // Endpoint Bing, Yandex, Google
        $ping_endpoints = [
            'google' => LBS_PING_ENDPOINTS['google'] . $encoded_sitemap,
            'bing'   => LBS_PING_ENDPOINTS['bing'] . $encoded_sitemap,
            'yandex' => LBS_PING_ENDPOINTS['yandex'] . $encoded_sitemap,
        ];

        // Baidu (se configurato)
        $baidu_token = get_option($this->option_prefix . 'baidu_token') ?: (defined('LEAN_BUNKER_BAIDU_TOKEN') ? LEAN_BUNKER_BAIDU_TOKEN : '');
        if ($baidu_token) {
            $ping_endpoints['baidu'] = "https://ziyuan.baidu.com/urls?site=" . urlencode(home_url()) . "&token=" . urlencode($baidu_token);
        }

        $results = [];
        $stats = get_option('lbs_ping_stats', []);

        foreach ($ping_endpoints as $engine => $url) {
            $response = wp_remote_get($url, [
                'timeout' => 8,
                'blocking' => true, // Blocking per ottenere lo status code
                'sslverify' => true,
                'user-agent' => 'LeanBunkerSEO/2.2.2'
            ]);

            if (is_wp_error($response)) {
                $results[$engine] = [
                    'status' => 'error',
                    'message' => $response->get_error_message(),
                    'http_code' => null
                ];
            } else {
                $http_code = wp_remote_retrieve_response_code($response);
                $body = wp_remote_retrieve_body($response);

                // Google: accetta sempre ma non garantisce nulla
                if ($engine === 'google') {
                    $results[$engine] = [
                        'status' => ($http_code === 200) ? 'success' : 'error',
                        'message' => 'Ping inviato (Google non garantisce l\'indicizzazione)',
                        'http_code' => $http_code
                    ];
                }
                // Bing, Yandex, Baidu: verificano effettivamente
                else {
                    $results[$engine] = [
                        'status' => ($http_code === 200) ? 'success' : 'error',
                        'message' => ($http_code === 200) ? 'Successo' : 'HTTP ' . $http_code,
                        'http_code' => $http_code,
                        'response' => substr($body, 0, 200) // Prime 200 caratteri per debug
                    ];
                }

                // Incrementa contatore ping
                if (!isset($stats[$engine])) {
                    $stats[$engine] = 0;
                }
                $stats[$engine]++;
            }
        }

        // Salva log
        update_option('lbs_last_ping_log', [
            'timestamp' => time(),
            'sitemap_url' => $sitemap_url,
            'results' => $results,
            'next_scheduled' => wp_next_scheduled('lean_bunker_sitemap_ping')
        ]);

        // Salva statistiche
        update_option('lbs_ping_stats', $stats);

        // Transiente per UI
        set_transient('lbs_sitemap_last_ping', time(), DAY_IN_SECONDS);
    }

    /**
    * Logga errori ping
    */
    private function log_ping_error($message) {
        $log = get_option('lbs_ping_error_log', []);
        $log[] = [
            'timestamp' => time(),
            'message' => $message
        ];

        // Mantieni solo ultimi 20 errori
        $log = array_slice($log, -20);
        update_option('lbs_ping_error_log', $log);
    }

    // ============================================================================
    // ROBOTS.TXT
    // ============================================================================

    /**
    * Aggiunge la sitemap al file robots.txt
    */
    public function add_sitemap_to_robots($output, $public) {
        if ($public && !str_contains($output, 'Sitemap:') && function_exists('home_url')) {
            $output .= "\nSitemap: " . home_url('/wp-sitemap.xml') . "\n";
        }
        return $output;
    }
}

// ============================================================================
// ISTANZIAZIONE & HOOKS
// ============================================================================

// Istanzia il plugin
$lean_bunker_seo = new Lean_Bunker_SEO();

// Hooks di attivazione/disattivazione FUORI dalla classe (best practice WordPress)
register_activation_hook(__FILE__, [$lean_bunker_seo, 'activate_pinger']);
register_deactivation_hook(__FILE__, [$lean_bunker_seo, 'deactivate_pinger']);