<?php
/**
 * Plugin Name: Lean Bunker Commerce
 * Plugin URI: https://leanbunker.it
 * Description: Configuratore di prodotti autonomo con modulo richiesta opzionale. Zero JS, single file, multisite-ready.
 * Version: 0.0.7
 * Author: Riccardo Bastillo
 * License: GPL-3.0+
 */

if (!defined('ABSPATH')) exit;

// === ADMIN MENU (per sito singolo) ===
add_action('admin_menu', 'lean_bunker_commerce_admin_menu');
function lean_bunker_commerce_admin_menu() {
    if (is_multisite()) {
        add_submenu_page(
            'options-general.php',
            'Lean Bunker Commerce',
            'Lean Bunker Commerce',
            'manage_options',
            'lean-bunker-commerce',
            'lean_bunker_commerce_settings_page'
        );
    } else {
        add_options_page(
            'Lean Bunker Commerce',
            'Lean Bunker Commerce',
            'manage_options',
            'lean-bunker-commerce',
            'lean_bunker_commerce_settings_page'
        );
    }
}

function lean_bunker_commerce_settings_page() {
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['lean_save'])) {
        check_admin_referer('lean_bunker_commerce_settings');
        $post_types = isset($_POST['post_types']) ? array_map('sanitize_key', $_POST['post_types']) : [];
        update_option('lean_bunker_commerce_post_types', $post_types);
        echo '<div class="notice notice-success"><p>Impostazioni salvate.</p></div>';
    }

    $enabled = get_option('lean_bunker_commerce_post_types', []);
    $all_post_types = get_post_types(['public' => true], 'objects');

    ?>
    <div class="wrap">
        <h1>Lean Bunker Commerce – Impostazioni</h1>
        <form method="post">
            <?php wp_nonce_field('lean_bunker_commerce_settings'); ?>
            <h2>Mostra metabox su questi tipi di contenuto:</h2>
            <table class="form-table">
                <?php foreach ($all_post_types as $name => $obj): ?>
                    <tr>
                        <th scope="row">
                            <label>
                                <input type="checkbox" name="post_types[]" value="<?php echo esc_attr($name); ?>" 
                                    <?php checked(in_array($name, $enabled)); ?> />
                                <?php echo esc_html($obj->label . ' (' . $name . ')'); ?>
                            </label>
                        </th>
                    </tr>
                <?php endforeach; ?>
            </table>
            <p class="submit">
                <input type="submit" name="lean_save" class="button-primary" value="Salva impostazioni" />
            </p>
        </form>
    </div>
    <?php
}

// === METABOX ===
function lean_bunker_commerce_get_enabled_post_types() {
    return (array) get_option('lean_bunker_commerce_post_types', []);
}

add_action('add_meta_boxes', 'lean_bunker_commerce_add_meta_box');
function lean_bunker_commerce_add_meta_box() {
    $enabled = lean_bunker_commerce_get_enabled_post_types();
    if (empty($enabled)) return;

    foreach ($enabled as $post_type) {
        add_meta_box(
            'lean_bunker_commerce_config',
            'Configuratore Prodotto – Lean Bunker Commerce',
            'lean_bunker_commerce_render_meta_box',
            $post_type,
            'normal'
        );
    }
}

function lean_bunker_commerce_render_meta_box($post) {
    wp_nonce_field('lean_bunker_commerce_save', 'lean_nonce');

    $config = get_post_meta($post->ID, '_lean_bunker_commerce_config', true);
    $enable_form = !empty($config['enable_form']);
    $fields_raw = '';

    if (!empty($config['fields']) && is_array($config['fields'])) {
        foreach ($config['fields'] as $name => $field) {
            $line = $name . ' | ' . ($field['label'] ?? '') . ' | ' . ($field['type'] ?? 'text');
            if ($field['type'] === 'select' && !empty($field['options'])) {
                $opts = [];
                foreach ($field['options'] as $label => $value) {
                    $opts[] = $label . '|' . $value;
                }
                $line .= ' | ' . implode(', ', $opts);
            }
            $fields_raw .= $line . "\n";
        }
    }

    echo '<p><strong>Campi configurabili</strong></p>';
    echo '<textarea name="lean_fields" style="width:100%;height:150px;" placeholder="nome | etichetta | tipo | opzioni (opzionale)">' . esc_textarea($fields_raw) . '</textarea>';
    echo '<p><small>Ogni riga = un campo.<br/>Tipi ammessi: <code>select</code>, <code>number</code>.<br/>Esempio: <code>pavimento | Tipo pavimento | select | Legno|80, Ceramica|45</code></small></p>';

    echo '<p><strong>Formula di calcolo</strong></p>';
    echo '<input type="text" name="lean_formula" value="' . esc_attr($config['formula'] ?? '') . '" style="width:100%;" />';
    echo '<p><small>Usa <code>{nome_campo}</code> per riferirti ai campi. Esempio: <code>{pavimento} * {mq}</code></small></p>';

    echo '<p><label><input type="checkbox" name="lean_enable_form" value="1" ' . checked($enable_form, true, false) . '> Abilita modulo richiesta dopo il calcolo</label></p>';

    echo '<hr style="margin:20px 0;">';
    echo '<h3>_shortcode per il frontend_</h3>';
    echo '<p>Per visualizzare il configuratore in una pagina o post, inserisci:</p>';
    echo '<p><code>[lean_calculator id="' . esc_attr($post->ID) . '"]</code></p>';
    echo '<p>Oppure, se usato nella stessa pagina del prodotto: <code>[lean_calculator]</code></p>';

    $example_url = wp_nonce_url(add_query_arg('lean_load_example', 1), 'lean_load_example');
    echo '<p><a href="' . esc_url($example_url) . '">UsageId esempio</a></p>';
}

add_action('admin_init', 'lean_bunker_commerce_handle_example');
function lean_bunker_commerce_handle_example() {
    if (isset($_GET['lean_load_example']) && wp_verify_nonce($_GET['_wpnonce'], 'lean_load_example')) {
        $_POST['lean_fields'] = "pavimento | Tipo pavimento | select | Legno|80, Ceramica|45, Marmo|120\nmq | Metri quadrati | number";
        $_POST['lean_formula'] = "{pavimento} * {mq}";
    }
}

add_action('save_post', 'lean_bunker_commerce_save_meta');
function lean_bunker_commerce_save_meta($post_id) {
    if (!isset($_POST['lean_nonce']) || !wp_verify_nonce($_POST['lean_nonce'], 'lean_bunker_commerce_save')) return;
    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
    if (!current_user_can('edit_post', $post_id)) return;

    $fields_raw = explode("\n", trim($_POST['lean_fields'] ?? ''));
    $fields = [];

    foreach ($fields_raw as $line) {
        $line = trim($line);
        if (!$line) continue;

        $parts = array_map('trim', explode('|', $line, 4));
        if (count($parts) < 3) continue;

        $name = sanitize_key($parts[0]);
        $label = sanitize_text_field($parts[1]);
        $type = in_array($parts[2], ['select', 'number']) ? $parts[2] : 'text';

        $field_data = ['label' => $label, 'type' => $type];

        if ($type === 'select' && isset($parts[3])) {
            $options = [];
            $opt_list = explode(',', $parts[3]);
            foreach ($opt_list as $opt) {
                $opt = trim($opt);
                if (!$opt) continue;
                $opt_parts = explode('|', $opt, 2);
                if (count($opt_parts) === 2) {
                    $opt_label = sanitize_text_field($opt_parts[0]);
                    $opt_value = floatval($opt_parts[1]);
                    $options[$opt_label] = $opt_value;
                }
            }
            $field_data['options'] = $options;
        }

        $fields[$name] = $field_data;
    }

    $config = [
        'fields' => $fields,
        'formula' => sanitize_text_field($_POST['lean_formula'] ?? ''),
        'enable_form' => !empty($_POST['lean_enable_form'])
    ];

    update_post_meta($post_id, '_lean_bunker_commerce_config', $config);
}

// === SHORTCODE FRONTEND (con modulo richiesta opzionale) ===
add_shortcode('lean_calculator', 'lean_bunker_commerce_render_frontend');
function lean_bunker_commerce_render_frontend($atts) {
    $atts = shortcode_atts(['id' => get_the_ID()], $atts);
    $post_id = intval($atts['id']);
    $config = get_post_meta($post_id, '_lean_bunker_commerce_config', true);

    if (empty($config['fields']) || !is_array($config['fields'])) {
        return '<p>Configuratore non disponibile.</p>';
    }

    $fields = $config['fields'];
    $formula = $config['formula'] ?? '';
    $enable_form = !empty($config['enable_form']);
    $calc_result = null;
    $user_values = [];

    // Gestione calcolo
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['lean_calc_submit'])) {
        foreach ($fields as $name => $field) {
            if ($field['type'] === 'select') {
                $user_values[$name] = floatval($_POST[$name] ?? 0);
            } elseif ($field['type'] === 'number') {
                $user_values[$name] = floatval($_POST[$name] ?? 0);
            }
        }

        $expr = $formula;
        foreach ($user_values as $var => $val) {
            $expr = str_replace('{' . $var . '}', $val, $expr);
        }

        if (preg_match('/^[0-9+\-*.\/\s\(\)]+$/', $expr)) {
            try {
                ob_start();
                $calc_result = eval("return ($expr);");
                ob_end_clean();
                if (!is_numeric($calc_result)) {
                    $calc_result = null;
                }
            } catch (ParseError $e) {
                $calc_result = null;
            }
        } else {
            $calc_result = null;
        }
    }

    // Gestione invio richiesta
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['lean_send_request'])) {
        $name = sanitize_text_field($_POST['lean_name'] ?? '');
        $email = sanitize_email($_POST['lean_email'] ?? '');
        $message = sanitize_textarea_field($_POST['lean_message'] ?? '');
        $total = floatval($_POST['lean_total'] ?? 0);
        $config_data = $_POST['lean_config'] ?? [];

        if (!$name || !$email || !is_email($email)) {
            return '<div class="notice notice-error"><p>Nome ed email sono obbligatori.</p></div>';
        }

        $to = get_option('admin_email');
        $subject = 'Nuova richiesta da ' . get_bloginfo('name');
        $body = "Hai ricevuto una nuova richiesta:\n\n";
        $body .= "Nome: $name\n";
        $body .= "Email: $email\n";
        $body .= "Totale: " . number_format($total, 2, ',', '.') . " €\n";
        $body .= "Prodotto: " . get_the_title($post_id) . "\n";
        $body .= "Configurazione:\n";
        foreach ($config_data as $key => $val) {
            $body .= "- $key: $val\n";
        }
        if ($message) {
            $body .= "\nMessaggio:\n$message";
        }

        $headers = ['Reply-To: ' . $email];
        if (wp_mail($to, $subject, $body, $headers)) {
            return '<div class="notice notice-success"><p>Grazie! La tua richiesta è stata inviata.</p></div>';
        } else {
            return '<div class="notice notice-error"><p>Errore nell’invio. Controlla la configurazione email del sito.</p></div>';
        }
    }

    // Render form
    $output = '<div class="lean-bunker-calculator">';
    $output .= '<form method="post" action="">';

    foreach ($fields as $name => $field) {
        $output .= '<div class="form-field">';
        $output .= '<label for="lean_' . esc_attr($name) . '">' . esc_html($field['label']) . ':</label><br/>';

        if ($field['type'] === 'select') {
            $output .= '<select id="lean_' . esc_attr($name) . '" name="' . esc_attr($name) . '" required class="regular-select">';
            foreach ($field['options'] as $label => $value) {
                $selected = (isset($_POST[$name]) && $_POST[$name] == $value) ? ' selected' : '';
                $output .= '<option value="' . esc_attr($value) . '"' . $selected . '>' . esc_html($label) . ' (' . esc_html($value) . ')</option>';
            }
            $output .= '</select>';
        } elseif ($field['type'] === 'number') {
            $val = isset($_POST[$name]) ? floatval($_POST[$name]) : 1;
            $output .= '<input type="number" id="lean_' . esc_attr($name) . '" name="' . esc_attr($name) . '" step="0.01" min="0" value="' . esc_attr($val) . '" required class="regular-text" />';
        }
        $output .= '</div>';
    }

    $output .= '<div class="submit">';
    $output .= '<button type="submit" name="lean_calc_submit" class="button button-primary">Calcola Totale</button>';
    $output .= '</div>';

    $output .= '</form>';

    // ✅ CORREZIONE FINALE: usa $calc_result (numero puro), non $result (stringa formattata)
    if ($calc_result !== null && is_numeric($calc_result)) {
        $formatted_result = number_format($calc_result, 2, ',', '.');
        $output .= '<div class="lean-result" style="margin-top:20px; padding:10px; background:#f9f9f9; border-left:4px solid #0073aa;"><strong>Totale: ' . esc_html($formatted_result) . ' €</strong></div>';

        if ($enable_form) {
            $output .= '<div class="lean-request-form" style="margin-top:20px;">';
            $output .= '<h3>Invia la tua richiesta</h3>';
            $output .= '<form method="post" action="">';
            $output .= '<div class="form-field"><label for="lean_name">Nome *</label><br/><input type="text" id="lean_name" name="lean_name" required class="regular-text" /></div>';
            $output .= '<div class="form-field"><label for="lean_email">Email *</label><br/><input type="email" id="lean_email" name="lean_email" required class="regular-text" /></div>';
            $output .= '<div class="form-field"><label for="lean_message">Messaggio</label><br/><textarea id="lean_message" name="lean_message" rows="3" class="regular-text"></textarea></div>';

            foreach ($user_values as $var => $val) {
                $output .= '<input type="hidden" name="lean_config[' . esc_attr($var) . ']" value="' . esc_attr($val) . '" />';
            }
            $output .= '<input type="hidden" name="lean_total" value="' . esc_attr($calc_result) . '" />';
            $output .= '<input type="hidden" name="lean_product_id" value="' . esc_attr($post_id) . '" />';

            $output .= '<div class="submit"><button type="submit" name="lean_send_request" class="button button-primary">Invia richiesta</button></div>';
            $output .= '</form>';
            $output .= '</div>';
        }
    }

    $output .= '</div>';

    return $output;
}